# PRESTOplay Media Player Package

## 1. Preface / Introduction

### a. Background and Purpose
This package provides a customizable, modular web-based media player built on top of [PRESTOplay](https://castlabs.com/products/prestoplay/). It supports advanced features such as DRM, adaptive streaming (DASH, HLS, Smooth), and robust subtitle and audio-only handling. The code is designed to quickly integrate secure video playback into their web applications, with options for custom controls and extensibility.

### b. Scope and Learning Objectives
- Understand the structure and flow of the player code.
- Learn how to configure and modify variables (such as video source, DRM credentials, and UI options) to achieve the desired playback.
- Gain insight into extending or customizing the player for your own needs.

---

## 2. Environment and Prerequisites

### a. Programming Language, Libraries, and Frameworks
- **Language:** JavaScript (ES6+)
- **Main Library:** [PRESTOplay](https://castlabs.com/prestoplay/web-apps/) (`@castlabs/prestoplay`)
- **Bundler:** Webpack

### b. Execution Environment
- **OS:** macOS, Windows, iOS, Android
- **Browser:** Modern browsers (Chrome, Edge, Safari)

### c. Installation and Setup
1. Download the zipped package at: [https://dr.orca.jp/PRESTOplay-drm-player/{uploaded-date}/prestoplay-package-zipped](https://dr.orca.jp/PRESTOplay-drm-player/{uploaded-date}/prestoplay-package-zipped)
2. Unzipped the package and run:
   ```sh
   npm run build
   npm link
   ```
3. Then navigate to your frontend project source, use this command to install the package:
   ```sh
   npm link prestoplay-drm-player
   ```

---

## 3. Overview of the Code

### a. Final Output
The final product is a JS package that initializes a video player that supports:
- DRM-protected and non-protected video playback
- Customizable controls (play, pause, subtitles, quality, etc.)
- Subtitle selection and rendering
- Responsive UI
- Audio-only content detection and UI adaptation

### b. Directory Structure
```
prestoplay-package/
  ├── demo.html                # Demo page for testing the player
  ├── src/
  │   ├── index.js             # Main player logic and class
  │   ├── custom-controls.js   # Custom UI controls orchestration
  │   ├── drm-config.js        # DRM configuration templates
  │   ├── audio-only-handler.js# Audio-only content handling
  │   ├── subtitle-handler.js  # Subtitle detection/handling logic
  │   ├── title-handler.js     # Title overlay handling
  │   ├── assets/
  │   │   └── icons/
  │   │       ├── index.js     # Icon exports
  │   │       └── *.svg        # Other UI icons
  │   ├── components/
  │   │   ├── play-pause/
  │   │   │   ├── play-pause.js # Play/pause button component
  │   │   │   └── play-pause.css # Play/pause styles
  │   │   ├── timeline/
  │   │   │   ├── timeline.js   # Timeline/progress bar component
  │   │   │   └── timeline.css  # Timeline styles
  │   │   ├── volume-control/
  │   │   │   ├── volume-control.js # Volume control component
  │   │   │   └── volume-control.css # Volume control styles
  │   │   ├── settings/
  │   │   │   ├── settings.js   # Main settings component
  │   │   │   ├── settings-popup/
  │   │   │   │   └── settings-popup.css # Settings popup styles
  │   │   │   ├── speed-control/
  │   │   │   │   ├── speed-control.js # Speed control component
  │   │   │   │   └── speed-control.css # Speed control styles
  │   │   │   ├── subtitle-control/
  │   │   │   │   ├── subtitle-control.js # Subtitle control component
  │   │   │   │   └── subtitle-control.css # Subtitle control styles
  │   │   │   └── quality-control/
  │   │   │       ├── quality-control.js # Quality control component
  │   │   │       └── quality-control.css # Quality control styles
  │   │   ├── fullscreen/
  │   │   │   ├── fullscreen.js # Fullscreen component
  │   │   │   └── fullscreen.css # Fullscreen styles
  │   │   ├── subtitle-toggle/
  │   │   │   ├── subtitle-toggle.js # Subtitle toggle component
  │   │   │   └── subtitle-toggle.css # Subtitle toggle styles
  │   │   ├── tooltip/
  │   │   │   ├── tooltip.js    # Tooltip component
  │   │   │   └── tooltip.css   # Tooltip styles
  │   │   ├── time-display/
  │   │   │   ├── time-display.js # Time display component
  │   │   │   └── time-display.css # Time display styles
  │   │   ├── title-overlay/
  │   │   │   └── title-overlay.css # Title overlay styles
  │   │   └── utils/
  │   │       └── button-factory.js # Button utility component
  │   └── styles/
  │       ├── index.css         # Main CSS entry point
  │       ├── main.css          # CSS imports orchestration
  │       ├── base.css          # Base container and video styles
  │       ├── controls.css      # Main controls container styles
  │       ├── fullscreen.css    # Fullscreen-specific styles
  │       └── responsive.css    # Responsive design styles
  ├── package.json              # Project metadata and dependencies
  ├── webpack.config.js         # Webpack build configuration
  └── README.md                 # Project documentation
```

**File Roles:**
- `index.js`: Main entry point, player class, and orchestration.
- `custom-controls.js`: Orchestrates all UI components and manages player controls.
- `drm-config.js`: DRM setup and configuration.
- `audio-only-handler.js`: Logic for handling audio-only content.
- `subtitle-handler.js`: Logic for subtitle detection and management.
- `title-handler.js`: Logic for title overlay display and management.
- `assets/icons/`: SVG icons for UI components.
- `components/`: Modular UI components (play-pause, timeline, volume, settings, etc.).
- `styles/`: Modular CSS architecture with component-specific styles.
- `demo.html`: Example usage and UI for testing.

**Architecture Overview:**
- **Modular Components**: Each UI element is a separate component with its own JS and CSS.
- **Component-Based CSS**: Styles are organized by component for better maintainability.
- **Icon Management**: SVG icons are externalized and imported as strings.
- **Container-Specific Storage**: Multiple player instances use container-specific storage to avoid conflicts.

---

## 4. Step-by-Step Code Explanation

### 4.1. Player Initialization
**Purpose:**
Create and configure a PRESTOplay player instance, attach it to the DOM, and set up controls.

**Example Initialization:**
```js
const player = new PrestoPlayDRMPlayer({
  containerId: "video-container", // (required) id of the container div
  source: "https://example.com/video.mpd", // (required) video source URL
  prestoplayCred: {                // (required) DRM credentials
    license: "<your-license>",   // (required)
    viewerId: "<your-viewer-id>" // (required)
  },
  width: 960,                     // (optional) video width, if omitted - 90% of mobile device's viewport, 60% for larger viewport
  height: 540,                    // (optional) video height, if omitted - will be auto-calculated by width * video's aspect ratio
  isDRM: true,                    // (optional) enable DRM
  drmConfig: {                    // (optional) DRM config object
    env: "DRMtoday_STAGING",      // (optional) DRM environment, "DRMtoday" (Production) or "DRMtoday_STAGING" (Staging)
    customData: {
      userId: "rental1",
      sessionId: "p0",
      merchant: "your-merchant-id",
      assetId: "Widevine-Test"
    }
  },
  autoplay: false,                // (optional)
  loop: false,                    // (optional)
  muted: false                    // (optional)
});
```
**Required Parameters:**
- `containerId`: The id of the HTML element to attach the player to.
- `source`: The video source URL (DASH, HLS, etc.).
- `prestoplayCred.license`: DRM license string.
- `prestoplayCred.viewerId`: DRM merchant UUID.

**Optional Parameters:**
- `width`, `height`: Video dimensions.
- `isDRM`: Enable DRM (default: false).
- `drmConfig`: DRM configuration object (see example above).
- `autoplay`, `loop`, `muted`: Standard video options.
- Other options of PRESTOplay can also be used here.

**Code Snippet (Class):**
```js
class PrestoPlayDRMPlayer {
  constructor(options = {}) {
    if (!options.containerId) throw new Error("containerId is required");
    if (!options.source) throw new Error("source is required");
    if (!options.prestoplayCred?.license && !options.prestoplayCred?.viewerId)
      throw new Error("license and viewerId is required to initialize a PRESTOplay player");
    // ...
  }
}
```

**About PRESTOplay License and Merchant:**
```js
this.player = new clpp.Player(this.video, this.options.prestoplayCred);
```
- `prestoplayCred.license` is the license string you obtain from the PRESTOplay license page. **Note:** This license is only valid for the domains that have been registered with the license itself. If you use it on an unregistered domain, playback will fail.
- `prestoplayCred.viewerId` is your merchant UUID (universally unique identifier) provided by PRESTOplay.

**Explanation:**
- The constructor validates required options and sets up the player container.
- Throws clear errors if required parameters are missing.

**Pitfall:**
- Forgetting to provide `containerId`, `source`, or PRESTOplay  license, DRM credentials will cause initialization to fail.

---

### 4.2. DRM and Source Loading
**Purpose:**
Configure DRM and load the video source.

**Code Snippet:**
```js
await this.player.load({
  source: this.options.source,
  autoplay: this.options.autoplay,
  loop: this.options.loop,
  muted: this.options.muted,
  drm: this.options.isDRM ? this.options.drmConfig : undefined,
});
```
**Explanation:**
- The `load` method loads the video, sets up the player, and applies DRM if needed.

**Pitfall:**
- DRM misconfiguration will result in playback errors.

---

### 4.3. Custom Controls and Subtitles
**Purpose:**
Add custom controls and handle subtitle tracks.

**Code Snippet:**
```js
import { createCustomControls } from "./custom-controls";
createCustomControls(this.player, this.video, this.options.containerId);
```
**Explanation:**
- Custom controls are modular and can be extended.
- Subtitle tracks are detected and managed via PRESTOplay APIs and the subtitle-handler module.

**Pitfall:**
- Not all browsers support all subtitle formats. Test with your target audience.

---

### 4.4. Audio-Only Content Handling
**Purpose:**
Detect and adjust UI for audio-only content.

**Code Snippet:**
```js
import { checkAudioOnlyContent } from "./audio-only-handler";
checkAudioOnlyContent(this);
```
**Explanation:**
- The function checks if the loaded media is audio-only and updates the UI accordingly.
- The function also hides the video container and only shows control bar if the media is audio-only.
- Uses the instance context (`this`) to access and update state.

**Pitfall:**
- If you customize the UI, ensure you handle the `data-audio-only` attribute correctly.

---

### 4.5. Subtitle Detection and Handling
**Purpose:**
Detect, enable, and manage subtitle tracks using a modular handler.

**Code Snippet:**
```js
import { setupSubtitleDetection } from "./subtitle-handler";
setupSubtitleDetection(this);
```
**Explanation:**
- The handler will periodically check for available subtitle tracks and enable them if found.
- Uses the instance context (`this`) to access and update state (`subtitleProcessed`, `prestoTracks`).

**Pitfall:**
- If you override subtitle logic, ensure you maintain correct state to avoid duplicate processing.

---

## 5. Use of Visuals and Diagrams

## 5.1. User Authorization Callback Implementation

<div style="text-align: center; margin: 20px 0;">
  <img src="user-authorization-callback.png" alt="User Authorization Callback Flow" style="max-width: 100%; height: auto; border: 1px solid #ddd; border-radius: 8px; box-shadow: 0 2px 8px rgba(0,0,0,0.1);">
</div>

**Authentication Flow:**

- **Authentication**: The user's Client device authenticates with Your session service.
- **Session ID Generation**: Upon successful authentication, Your session service generates and sends a unique sessionId back to the Client device.
- **License Request**: The Client device sends a license request to the DRMtoday license server. This request includes the sessionId in its custom data payload.
- **Authorization Callback**: The DRMtoday license server pauses processing and sends a real-time "User authorization callback" to Your auth service, forwarding the sessionId.
- **Session Validation**: Your auth service validates the sessionId to confirm the user is active and authorized to view the content.
- **Customer Rights Token**: After successful validation, Your auth service sends a Customer rights token (containing playback rules) back to the DRMtoday license server.
- **License Creation**: DRMtoday uses the Customer Rights Token to create the final, secure license.
- **License Delivery**: The DRMtoday license server sends the final license back to the Client device, which can then decrypt and play the video.

### 5.2. Authorization Methods Comparison

| Feature | Upfront Authorization Token | User Authorization Callback |
|---|---|---|
| **Core Idea** | The backend server pre-authorizes the user and gives the player a "ticket" (the token) to present to DRMtoday. | DRMtoday receives a request and asks the backend server in real-time, "Is this user allowed to watch this?" |
| **Workflow** | One-way trip: Client -> The Backend Server -> Client -> DRMtoday | Round trip: Client -> DRMtoday -> The Backend Server -> DRMtoday -> Client |
| **Backend Responsibility** | Generate a secure, signed, and short-lived token before the license request is made. | Implement a server endpoint that can receive a real-time callback from DRMtoday and respond instantly. |
| **Complexity** | Simpler for the DRM provider (DRMtoday), but requires the backend server to handle token generation logic. | More complex backend logic is required to handle the real-time callback, but it gives you more direct control. |
| **Latency** | Potentially lower latency, as DRMtoday can issue the license immediately after validating the token without an extra network hop. | Slightly higher latency due to the additional real-time callback from DRMtoday to the backend server. |
| **Real-time Control** | Less real-time control. A user's rights are based on when the token was generated (tokens are valid for 10 minutes). | Maximum real-time control. You can deny a license request the instant a user's subscription expires. |
| **Best For** | Standard VOD playback, SPAs, and situations where you want to reduce the real-time load on the backend server. | Live streaming, pay-per-view events, or systems where immediate, up-to-the-second authorization is critical. |

---

## 6. Supplementary Notes and Advanced Topics

### a. Better Design Practices
- **Separation of Concerns:**
  - Logic for controls, DRM, audio-only, and subtitle handling are split into separate modules for maintainability.
- **Extensibility:**
  - You can add new controls or features by extending the relevant modules.
- **Context Passing:**
  - All handler modules receive the player instance (`this`) as context, making state management explicit and modular.

### b. Advanced Considerations
- **Security:**
  - Never expose DRM credentials in client-side code for production.
  - Use HTTPS for all media and license requests.

---

## 7. Error Handling and Debugging Tips

### a. Common Error Messages
- **"Container with id ... not found":**
  - Check your HTML and ensure the container exists.
- **"license and viewerId is required":**
  - Provide valid DRM credentials.
- **DRM playback errors:**
  - Check your license server, network, and browser DRM support.

### b. Debugging Tools
- **Browser DevTools:**
  - Use the Console and Network tabs to inspect errors and license requests.
- **PRESTOplay Logs:**
  - Enable verbose logging for more details.
- **Custom Log Function:**
  - Use the provided `log()` utility in `demo.html` for step-by-step tracing.

---