import Loader from './../Loaders/Loader';
import { VideoSegment } from './VideoSegment';
/**
 * @class
 * @description Class that stores downloaded segments so they can be shared.
 * It automatically removes old and unused segments when needed.
 * @exports SegmentStorage
 */
export default class SegmentStorage {
    private readonly _localMem;
    private readonly _loader;
    private _maxSize;
    private _expTime;
    private _lastSegmentId;
    /**
     * Constructs Storage object.
     * @param {Loader} loader Loader object.
     */
    constructor(loader: Loader);
    /**
     * Set the options of the storage, using API response.
     * @param {balancerResponse} e API response
     * @public
     */
    setSettings(e: balancerResponse): void;
    getMaxSize(): number;
    /**
     * Checks in the storage for the requested segment and returns it. Undefined if is not there.
     * When returned updates the useTime variable of the segment so it wont be auto-deleted.
     * @param {string} id Id of the segment.
     * @returns {VideoSegment|undefined} Segment object corresponding to the id, or undefined.
     * @public
     */
    getSegment(id: string): VideoSegment | undefined;
    /**
     * Checks in the storage for the requested segment and returns if its there or not.
     * @param {string} id Id of the segment.
     * @returns {boolean} Availability of the chosen segment.
     * @private
     */
    hasSegment(id: string): boolean;
    /**
     * Stores the given segment.
     * If size exceeds the maximum it will delete the oldest segment.
     * @param {VideoSegment} newSegment Segment object to store.
     * @param {ArrayBuffer} data Array buffer with the data of the segment.
     * @param {boolean} update If it should replace previous values, false by default.
     * @public
     */
    storeSegment(segment: VideoSegment): boolean;
    /**
     * Calls recursively every second to check for unused/old segments and delete them.
     * @private
     */
    private _removeOldSegments;
    /**
     * Checks for the memory size (number of segments) and deletes oldest ones if size exceeded the maximum.
     * @private
     */
    private _reduceMemSize;
    /**
     * Generates a map for peers. An object with key name equal to manifest url, an object as a value.
     * This array contains as keys the path names of the video segments and as value its status
     * (0 downloaded, 1 downloading)
     * @returns {map} Map of the available chunks to share with peers.
     * @public
     */
    createMap(): map;
    destroy(): void;
    getSize(): number;
    getLastSegmentId(): string;
    getLastSegmentCreatedAt(): number;
}
