import { SessionListeners } from '../common/Constants';
import { Service } from '../core/nqs/Services';
/**
 * Session is the generic class to handle Session management for all the applications.
 * Every plugin will have an instance.
 */
export default class Session {
    private static BEAT_INTERVAL;
    private core;
    private coreStorage;
    private options;
    private accountCode;
    private requestHandler;
    private requestBuilder;
    _beat: any;
    _lastNavigation: {
        page: string;
        route: string;
    };
    _registeredProperties: {
        dimensions: object;
        values: object;
    } | null;
    private listeners;
    private expirationManager;
    /**
     * Constructs the Sessions class
     */
    constructor(accountCode: string);
    addListener(eventName: SessionListeners, callback: (...args: any[]) => void): void;
    private fireListener;
    /**
     * Starts a new session. If a session exists, stops it and starts a new one.
     *
     * @param options - Object of key:value options (optional).
     * @param dimensions - Object of key:value params to add to the request (optional).
     * @param onSuccess - Optional callback function to be executed if the session start is successful.
     * @param onFail - Optional callback function to be executed if the session start fails.
     * @param forceStart - Optional parameter to force sending the /start NQS event
     */
    start(options?: object, dimensions?: object, onSuccess?: () => void, onFail?: () => void, forceStart?: boolean): void;
    /**
     * Stop session
     * @param params
     * @param onSuccess
     * @param onFail
     * @param forceStop
     * @param sessionStarted
     * @returns
     */
    stop(params?: any, onSuccess?: () => void, onFail?: () => void, forceStop?: boolean, sessionStarted?: boolean): void;
    /**
     * Sends session event
     * @param eventName
     * @param dimensions
     * @param values
     * @param topLevelDimensions
     * @param onSuccess
     * @param onFail
     * @returns
     */
    sendEvent(eventName?: string, dimensions?: object, values?: object, topLevelDimensions?: object, onSuccess?: () => void, onFail?: () => void): void;
    /**
     * Emits session start request.
     *
     * @param dimensions - Object of key:value params to add to the request.
     * @param onSuccess - Optional callback function to be executed if the request is successful.
     * @param onFail - Optional callback function to be executed if the request fails.
     */
    fireNav(dimensions?: object, onSuccess?: () => void, onFail?: () => void): void;
    /**
     * Destroy session object
     */
    destroy(): void;
    /**
     * Gets Analytics options. See {@link Options}.
     */
    getOptions(): any;
    isUsingLegacy(): boolean;
    refreshSessionToken(): void;
    /**
     * Get last activity timestamp
     */
    _setLastActive(): void;
    protected getAccountCode(): string;
    /** Register properties sent by the User, to send in all the events
     *
     * @param dimensions - Object of key:value dimensions.
     * @param values - Object of key:value values.
     */
    register(dimensions?: object, values?: object): void;
    /** Calls register if registeredProperties is empty
     *
     * @param dimensions - Object of key:value dimensions.
     * @param values - Object of key:value values.
     */
    registerOnce(dimensions: object, values: object): void;
    /** Unregister all properties registered with register() */
    unregister(): void;
    /**
     * Splits params in dimensions (strings) and values (numbers)
     *
     * @param dimensions - Object of key:value dimensions to split before adding to request.
     * @param values - Object of key:value values to split before adding to request.
     * @param eventName - Event name.
     * @param isNavigation - Boolean to check if it's a navigation event.
     * @param isStart - Boolean to check if it's a start event.
     */
    private _getParamsJson;
    /**
     * Checks whether the session is active or not
     * @returns
     */
    isActive(): boolean;
    /**
     * Send a session request
     * @param service
     * @param params
     * @param onSuccess
     * @param onFail
     * @private
     */
    private _sendSession;
    /**
     * Adds a listener for analytics request events.
     * @param listener A function of type `(serviceName: string, params: Map<string, string>) => void` to be called when an analytics request is about to be sent.
     */
    addOnWillSendRequestListener(listener: (serviceName: string, params: Map<string, string>) => void): void;
    /**
     * Removes a previously added listener for analytics request events.
     * @param listener The listener function of type `(serviceName: string, params: Map<string, string>) => void` to remove from the list of analytics request listeners.
     */
    removeOnWillSendRequestListener(listener: (serviceName: string, params: Map<string, string>) => void): void;
    /**
     * Process Internal Params object
     * @param params
     * @returns {{}|*|{}}
     * @private
     */
    private _parseInternalParams;
    /**
     * Sends beat request
     *
     * @param diffTime - Time since the last ping
     *
     * @private
     */
    _sendBeat(diffTime: number): void;
    /**
     * Check if Session is expired
     * If is true, restart the session automatically
     *
     * @returns
     */
    expiredSession(dimensions?: any, sessionStarted?: boolean): boolean;
    /**
     * Check if change page/route (to avoid duplicated navigations)
     * @param params
     * @returns {boolean}
     * @private
     */
    private _checkDifferentNavigation;
    /**
     * Returns a json with the metrics to be reported in beats when changed
     *
     */
    getSessionMetrics(): object;
    /**
     * Is post method enabled
     * @returns {boolean}
     */
    isMethodPostEnabled(): boolean;
    getSessionRoot(): string;
    getSessionHost(): string;
    /**
     *
     * @param willSendLog
     * @param service
     * @param params
     */
    _sendPluginLogs(willSendLog: string, service: Service, params: object): void;
    _logFireSessionStartEvent(dimensions: object): void;
    private _logFireSessionStopEvent;
    private _logFireEventListener;
    private _logFireNavListener;
}
