import { fullScreenIcon, exitFullScreenIcon } from "../../assets/icons/index.js";

export function createFullscreenComponent(container, video, isAudioOnly = false) {
  if (isAudioOnly) {
    // Return null for audio-only content
    return null;
  }

  // Create fullscreen button
  const fullscreenBtn = document.createElement("button");
  fullscreenBtn.className = "pp-fs-btn";
  fullscreenBtn.title = "Fullscreen";
  fullscreenBtn.innerHTML = fullScreenIcon;

  // Fullscreen state variables
  let fullscreenHideTimeout;
  let isInFullscreen = false;
  let lastMousePosition = { x: 0, y: 0 };

  // Function to update fullscreen icon
  function updateFullscreenIcon() {
    const isFullscreen = !!document.fullscreenElement;
    fullscreenBtn.innerHTML = isFullscreen
      ? exitFullScreenIcon
      : fullScreenIcon;
    fullscreenBtn.title = isFullscreen ? "Exit Fullscreen" : "Fullscreen";
  }

  // Fullscreen toggle functionality
  fullscreenBtn.addEventListener("click", () => {
    if (!document.fullscreenElement) {
      container.requestFullscreen();
    } else {
      document.exitFullscreen();
    }
  });

  // Update icon when fullscreen state changes
  document.addEventListener("fullscreenchange", updateFullscreenIcon);
  document.addEventListener("webkitfullscreenchange", updateFullscreenIcon);
  document.addEventListener("mozfullscreenchange", updateFullscreenIcon);
  document.addEventListener("MSFullscreenChange", updateFullscreenIcon);

  // Control bar visibility functions (will be passed from parent)
  let showControlBar, hideControlBar;

  // Fullscreen auto-hide logic
  function startFullscreenAutoHide() {
    if (fullscreenHideTimeout) {
      clearTimeout(fullscreenHideTimeout);
    }
    fullscreenHideTimeout = setTimeout(() => {
      if (isInFullscreen && !document.fullscreenElement) {
        isInFullscreen = false;
        return;
      }
      if (isInFullscreen) {
        // Check if mouse is currently over control bar or title overlay
        const controlsContainer = container.querySelector(
          ".prestoplay-custom-controls"
        );
        const titleOverlay = container.querySelector(".pp-title-overlay");

        // Get current mouse position
        const mouseX = lastMousePosition.x;
        const mouseY = lastMousePosition.y;

        // Check if mouse is over control bar
        let isOverControls = false;
        if (controlsContainer) {
          const rect = controlsContainer.getBoundingClientRect();
          isOverControls =
            mouseX >= rect.left &&
            mouseX <= rect.right &&
            mouseY >= rect.top &&
            mouseY <= rect.bottom;
        }

        // Check if mouse is over title overlay
        let isOverTitle = false;
        if (titleOverlay) {
          const rect = titleOverlay.getBoundingClientRect();
          isOverTitle =
            mouseX >= rect.left &&
            mouseX <= rect.right &&
            mouseY >= rect.top &&
            mouseY <= rect.bottom;
        }

        // Only hide if mouse is not over control areas
        if (!isOverControls && !isOverTitle) {
          if (hideControlBar) {
            hideControlBar();
          }
          // Also hide title overlay
          if (window.hideTitleOverlay) {
            window.hideTitleOverlay();
          }
        } else {
          // Mouse is over control areas, restart timer
          startFullscreenAutoHide();
        }
      }
    }, 1500); // 1.5 seconds
  }

  function stopFullscreenAutoHide() {
    if (fullscreenHideTimeout) {
      clearTimeout(fullscreenHideTimeout);
      fullscreenHideTimeout = null;
    }
  }

  function handleMouseMove(e) {
    if (!isInFullscreen) return;

    const currentPosition = { x: e.clientX, y: e.clientY };
    const distance = Math.sqrt(
      Math.pow(currentPosition.x - lastMousePosition.x, 2) +
        Math.pow(currentPosition.y - lastMousePosition.y, 2)
    );

    // Only trigger if mouse moved more than 5px (to avoid tiny movements)
    if (distance > 5) {
      // Show controls when mouse moves anywhere in fullscreen
      if (showControlBar) {
        showControlBar();
      }
      // Also show title overlay
      if (window.showTitleOverlay) {
        window.showTitleOverlay();
      }
      startFullscreenAutoHide();
      lastMousePosition = currentPosition;

      // Share mouse position with title overlay
      window.lastMousePosition = currentPosition;
    }
  }

  function handleFullscreenChange() {
    isInFullscreen = !!document.fullscreenElement;

    if (isInFullscreen) {
      // Show controls initially in fullscreen
      if (showControlBar) {
        showControlBar();
      }
      // Also show title overlay
      if (window.showTitleOverlay) {
        window.showTitleOverlay();
      }
      startFullscreenAutoHide();
      // Add mouse move listener for fullscreen
      document.addEventListener("mousemove", handleMouseMove);
    } else {
      // Exit fullscreen - remove listeners and show controls normally
      stopFullscreenAutoHide();
      document.removeEventListener("mousemove", handleMouseMove);
      if (showControlBar) {
        showControlBar();
      }
      // Also show title overlay
      if (window.showTitleOverlay) {
        window.showTitleOverlay();
      }
    }
  }

  // Listen for fullscreen changes
  document.addEventListener("fullscreenchange", handleFullscreenChange);
  document.addEventListener("webkitfullscreenchange", handleFullscreenChange);
  document.addEventListener("mozfullscreenchange", handleFullscreenChange);
  document.addEventListener("MSFullscreenChange", handleFullscreenChange);

  // Expose fullscreen functions for external access
  window.isInFullscreen = () => isInFullscreen;
  window.startFullscreenAutoHide = startFullscreenAutoHide;

  // Function to set control bar functions (called from parent)
  function setControlBarFunctions(showFn, hideFn) {
    showControlBar = showFn;
    hideControlBar = hideFn;
  }

  return {
    button: fullscreenBtn,
    setControlBarFunctions,
    isInFullscreen: () => isInFullscreen,
    startFullscreenAutoHide,
    stopFullscreenAutoHide,
    handleMouseMove,
    handleFullscreenChange,
  };
} 