import { loadingIcon } from "../../assets/icons/index.js";
import "./loading.css";

export function createLoadingComponent(video, container) {
  // Find video wrapper (the div that contains the video element)
  const videoWrapper = video.parentElement;
  
  // Create loading overlay
  const loadingOverlay = document.createElement("div");
  loadingOverlay.className = "pp-loading-overlay";

  // Create loading spinner
  const loadingSpinner = document.createElement("div");
  loadingSpinner.className = "pp-loading-spinner";
  loadingSpinner.innerHTML = loadingIcon;

  loadingOverlay.appendChild(loadingSpinner);
  videoWrapper.appendChild(loadingOverlay);

  // Prevent clicks on video when loading
  function preventVideoClicks(e) {
    e.preventDefault();
    e.stopPropagation();
    return false;
  }

  // Show/hide loading functions
  function showLoading() {
    loadingOverlay.style.display = "flex";
    // Prevent video clicks when loading
    video.addEventListener("click", preventVideoClicks, true);
    videoWrapper.addEventListener("click", preventVideoClicks, true);
  }

  function hideLoading() {
    loadingOverlay.style.display = "none";
    // Re-enable video clicks when not loading
    video.removeEventListener("click", preventVideoClicks, true);
    videoWrapper.removeEventListener("click", preventVideoClicks, true);
  }

  // Event listeners for loading states
  function handleLoadStart() {
    showLoading();
  }

  function handleCanPlay() {
    hideLoading();
  }

  function handleSeeking() {
    showLoading();
  }

  function handleSeeked() {
    hideLoading();
  }

  function handleWaiting() {
    showLoading();
  }

  function handlePlaying() {
    hideLoading();
  }

  function handlePause() {
    hideLoading();
  }

  function handleEnded() {
    hideLoading();
  }

  function handleError() {
    hideLoading();
  }

  // Add event listeners
  video.addEventListener("loadstart", handleLoadStart);
  video.addEventListener("canplay", handleCanPlay);
  video.addEventListener("seeking", handleSeeking);
  video.addEventListener("seeked", handleSeeked);
  video.addEventListener("waiting", handleWaiting);
  video.addEventListener("playing", handlePlaying);
  video.addEventListener("pause", handlePause);
  video.addEventListener("ended", handleEnded);
  video.addEventListener("error", handleError);

  // Cleanup function
  function cleanup() {
    video.removeEventListener("loadstart", handleLoadStart);
    video.removeEventListener("canplay", handleCanPlay);
    video.removeEventListener("seeking", handleSeeking);
    video.removeEventListener("seeked", handleSeeked);
    video.removeEventListener("waiting", handleWaiting);
    video.removeEventListener("playing", handlePlaying);
    video.removeEventListener("pause", handlePause);
    video.removeEventListener("ended", handleEnded);
    video.removeEventListener("error", handleError);
    
    // Remove click prevention listeners
    video.removeEventListener("click", preventVideoClicks, true);
    videoWrapper.removeEventListener("click", preventVideoClicks, true);
    
    if (loadingOverlay.parentNode) {
      loadingOverlay.parentNode.removeChild(loadingOverlay);
    }
  }

  return {
    show: showLoading,
    hide: hideLoading,
    cleanup
  };
} 