// Quality Control Component
import { qualityIcon, chevronLeftIcon } from "../../../assets/icons/index.js";

export function createQualityControl(player, settingsPopup) {
  // Quality item
  const qualityItem = document.createElement("div");
  qualityItem.className = "pp-settings-item";
  qualityItem.innerHTML = `
    <div style="display: flex; align-items: center; gap: 12px;">
        ${chevronLeftIcon}
      <div style="width: 20px; height: 20px; display: flex; align-items: center; justify-content: center;">
        ${qualityIcon}
      </div>
      <span>Quality</span>
    </div>
    <div style="display: flex; align-items: center; gap: 8px;">
      <span class="pp-quality-current">Auto</span>
    </div>
  `;

  // Quality submenu
  const qualitySubmenu = document.createElement("div");
  qualitySubmenu.className = "pp-quality-submenu pp-display-none";

  // Auto quality option
  const autoItem = document.createElement("div");
  autoItem.className = "pp-quality-option";
  autoItem.innerHTML = `
    <span>Auto</span>
    <span class="pp-quality-checkmark">✓</span>
  `;
  autoItem.querySelector(".pp-quality-checkmark").style.color = "#2196F3";

  // Store quality options for selection highlighting
  const qualityOptions = [autoItem];

  function convertVideoResolution(width, height) {
    if (!width || !height || width <= 0 || height <= 0) {
      return "Unknown";
    }

    return `${height}p`;
  }

  // Function to add video tracks
  function addVideoTracks(videoTracks) {
    // Clear existing quality options (except Auto)
    qualityOptions.slice(1).forEach(option => option.remove());
    qualityOptions.splice(1);

    if (!videoTracks || videoTracks.length === 0) {
      return;
    }

    // Collect all renditions from all tracks
    const allRenditions = [];
    videoTracks.forEach((track) => {
      if (track.renditions?.length > 0) {
        track.renditions.forEach((rendition) => {
          allRenditions.push(rendition);
        });
      } else {
        // Handle tracks without renditions
        allRenditions.push(track);
      }
    });

    // Group renditions by resolution and keep only the highest bandwidth/quality for each
    const resolutionMap = new Map();
    allRenditions.forEach((rendition) => {
      const resolution = convertVideoResolution(
        rendition.width,
        rendition.height
      );
      const bandwidth = rendition.bandwidth || rendition.bitrate || 0;
      const height = rendition.height || 0;

      if (!resolutionMap.has(resolution)) {
        resolutionMap.set(resolution, {
          rendition,
          bandwidth,
          height,
          resolution,
        });
      } else {
        const existing = resolutionMap.get(resolution);
        // Keep the one with higher bandwidth, or higher height if bandwidth is equal
        if (
          bandwidth > existing.bandwidth ||
          (bandwidth === existing.bandwidth && height > existing.height)
        ) {
          resolutionMap.set(resolution, {
            rendition,
            bandwidth,
            height,
            resolution,
          });
        }
      }
    });

    // Sort by resolution height in descending order (highest quality first)
    const sortedQualities = Array.from(resolutionMap.values()).sort((a, b) => {
      const heightA = a.height || 0;
      const heightB = b.height || 0;
      return heightB - heightA;
    });

    // Create quality options for each unique resolution
    sortedQualities.forEach((qualityData) => {
      const { rendition, resolution } = qualityData;
      const label = resolution; // No bandwidth display as requested

      const option = document.createElement("div");
      option.className = "pp-quality-option";
      option.innerHTML = `
        <span>${label}</span>
        <span class="pp-quality-checkmark" style="visibility: hidden;">✓</span>
      `;

      option.addEventListener("click", () => {
        if (player?.getTrackManager) {
          const trackManager = player.getTrackManager();
          if (trackManager) {
            try {
              // First disable ABR
              if (player.configure) {
                player.configure({
                  abr: { enabled: false },
                });
              }

              // Try to find the rendition using findVideoRendition method
              if (trackManager.findVideoRendition) {
                const foundRendition = trackManager.findVideoRendition({
                  id: rendition.id,
                  width: rendition.width,
                  height: rendition.height,
                  bandwidth: rendition.bandwidth,
                });

                if (foundRendition) {
                  try {
                    trackManager.setVideoRendition(foundRendition);
                  } catch (e) {
                    console.error(`Failed to set found rendition: ${e.message}`);
                  }
                }
              }

              // If findVideoRendition didn't work, try the original approach
              if (!trackManager.findVideoRendition) {
                try {
                  trackManager.setVideoRendition(rendition);
                } catch (e) {
                  console.error(`Original setVideoRendition failed: ${e.message}`);

                  // Last resort: try with just the ID
                  if (trackManager.setVideoRendition && rendition.id) {
                    try {
                      trackManager.setVideoRendition(rendition.id);
                    } catch (e2) {
                      console.error(`Even ID-only approach failed: ${e2.message}`);
                    }
                  }
                }
              }

              // Update UI
              updateQualitySelection(label, option);
            } catch (e) {
              console.error(`Failed to set rendition: ${e.message}`);
            }
          }
        }
      });

      qualityOptions.push(option);
      qualitySubmenu.appendChild(option);
    });
  }

  autoItem.addEventListener("click", () => {
    if (player?.getTrackManager) {
      const trackManager = player.getTrackManager();
      if (trackManager) {
        try {
          // First, re-enable ABR for automatic quality selection
          if (player.configure) {
            player.configure({
              abr: { enabled: true },
            });
          }

          // Set video rendition to null to let ABR take over
          if (trackManager.setVideoRendition) {
            trackManager.setVideoRendition(null);
          }
        } catch (e) {
          console.error("Failed to set auto quality:", e.message);
        }
      }
    }

    // Update UI
    updateQualitySelection("Auto", autoItem);
  });

  qualitySubmenu.appendChild(autoItem);

  // Function to update quality selection
  function updateQualitySelection(qualityLabel, selectedOption) {
    // Update main quality display
    const currentQuality = qualityItem.querySelector(".pp-quality-current");
    currentQuality.textContent = qualityLabel;

    // Clear all selections
    qualityOptions.forEach((option) => {
      option.classList.remove("selected");
      const checkmark = option.querySelector(".pp-quality-checkmark");
      if (checkmark) {
        checkmark.style.visibility = "hidden";
        checkmark.style.color = "#fff";
      }
    });

    // Highlight selected option
    if (selectedOption) {
      selectedOption.classList.add("selected");
      const checkmark = selectedOption.querySelector(".pp-quality-checkmark");
      if (checkmark) {
        checkmark.style.visibility = "visible";
        checkmark.style.color = "#2196F3";
      }
    }
  }

  // Quality item click handler
  qualityItem.addEventListener("click", (e) => {
    e.stopPropagation();
    // Hide other submenus first
    const speedSubmenu = settingsPopup.querySelector(".pp-speed-submenu");
    if (speedSubmenu) {
      speedSubmenu.classList.add("pp-display-none");
      speedSubmenu.classList.remove("pp-display-block");
    }
    const subtitleSubmenu = settingsPopup.querySelector(".pp-subtitle-submenu");
    if (subtitleSubmenu) {
      subtitleSubmenu.classList.add("pp-display-none");
      subtitleSubmenu.classList.remove("pp-display-block");
    }
    
    if (qualitySubmenu.classList.contains("pp-display-none")) {
      qualitySubmenu.classList.remove("pp-display-none");
      qualitySubmenu.classList.add("pp-display-block");
    } else {
      qualitySubmenu.classList.add("pp-display-none");
      qualitySubmenu.classList.remove("pp-display-block");
    }
  });

  // Close submenu when clicking outside
  document.addEventListener("click", (e) => {
    if (!qualitySubmenu.contains(e.target) && !qualityItem.contains(e.target)) {
      qualitySubmenu.classList.add("pp-display-none");
      qualitySubmenu.classList.remove("pp-display-block");
    }
    // Also close other submenus
    const speedSubmenu = settingsPopup.querySelector(".pp-speed-submenu");
    if (
      speedSubmenu &&
      !speedSubmenu.contains(e.target) &&
      !settingsPopup.querySelector(".pp-settings-item:nth-child(1)").contains(e.target)
    ) {
      speedSubmenu.classList.add("pp-display-none");
      speedSubmenu.classList.remove("pp-display-block");
    }
    const subtitleSubmenu = settingsPopup.querySelector(".pp-subtitle-submenu");
    if (
      subtitleSubmenu &&
      !subtitleSubmenu.contains(e.target) &&
      !settingsPopup.querySelector(".pp-settings-item:nth-child(2)").contains(e.target)
    ) {
      subtitleSubmenu.classList.add("pp-display-none");
      subtitleSubmenu.classList.remove("pp-display-block");
    }
  });

  // Initialize with auto quality selected
  updateQualitySelection("Auto", autoItem);

  return {
    item: qualityItem,
    submenu: qualitySubmenu,
    addVideoTracks,
    updateQualitySelection,
  };
} 