// Subtitle Control Component
import { subtitleIcon, chevronLeftIcon } from "../../../assets/icons/index.js";

export function createSubtitleControl(player, settingsPopup) {
  // Subtitle item
  const subtitleItem = document.createElement("div");
  subtitleItem.className = "pp-settings-item";
  subtitleItem.innerHTML = `
    <div style="display: flex; align-items: center; gap: 12px;">
      ${chevronLeftIcon}
      <div style="width: 20px; height: 20px; display: flex; align-items: center; justify-content: center;">
        ${subtitleIcon}
      </div>
      <span>Subtitles/CC</span>
    </div>
    <div style="display: flex; align-items: center; gap: 8px;">
      <span class="pp-subtitle-current">Off</span>
    </div>
  `;

  // Subtitle submenu
  const subtitleSubmenu = document.createElement("div");
  subtitleSubmenu.className = "pp-subtitle-submenu pp-display-none";

  // Default "Off" option
  const subtitleOffItem = document.createElement("div");
  subtitleOffItem.id = "subtitle-off";
  subtitleOffItem.className = "pp-subtitle-item";
  subtitleOffItem.innerHTML = `
    <span>Off</span>
    <span class="pp-subtitle-checkmark" style="visibility: visible; color: #2196F3;">✓</span>
  `;

  // Store subtitle items for selection highlighting
  const subtitleItems = [subtitleOffItem]; // The "Off" item

  // Function to add subtitle tracks
  function addSubtitleTracks(prestoTracks) {
    // Clear existing track items (except "Off")
    subtitleItems.slice(1).forEach((item) => item.remove());
    subtitleItems.splice(1);

    if (!prestoTracks || prestoTracks.length === 0) {
      return;
    }

    // Add track options
    prestoTracks.forEach((track, idx) => {
      const trackLabel = track.label || track.language || `Track ${idx + 1}`;

      const item = document.createElement("div");
      item.id = track.id;
      item.className = "pp-subtitle-item pp-subtitle-track-item";
      item.innerHTML = `
        <span>${trackLabel}</span>
        <span class="pp-subtitle-checkmark" style="visibility: hidden; color: #2196F3;">✓</span>
      `;

      item.addEventListener("click", (e) => {
        e.stopPropagation();
        // Use PRESTOplay TrackManager to enable subtitles
        if (player?.getTrackManager) {
          const trackManager = player.getTrackManager();
          if (trackManager?.setTextTrack) {
            try {
              // First, let's try to get all available tracks to find the correct one
              const availableTracks = trackManager.getTextTracks();

              // Find the exact track match - prioritize ID match first
              let targetTrack = null;
              let trackIndex = -1;

              // First try to find by exact ID match
              trackIndex = availableTracks.findIndex((t) => t.id === track.id);
              if (trackIndex >= 0) {
                targetTrack = availableTracks[trackIndex];
              } else {
                // If no ID match, try by language
                trackIndex = availableTracks.findIndex(
                  (t) => t.language === track.language
                );
                if (trackIndex >= 0) {
                  targetTrack = availableTracks[trackIndex];
                } else {
                  return;
                }
              }

              // Set the exact track we found
              trackManager.setTextTrack(targetTrack);
            } catch (e) {
              console.error(`Failed to set PRESTOplay track: ${e}`);
            }
          }
        }

        // Update selection highlighting
        updateSelectionHighlighting();

        // Hide all submenus
        subtitleSubmenu.classList.add("pp-display-none");
        subtitleSubmenu.classList.remove("pp-display-block");

        updateSubtitleItemState(true);

        // Update subtitle toggle button state after a short delay
        setTimeout(() => {
          const containerId = settingsPopup.closest('.clpp-container')?.id;
          if (containerId && window.updateSubtitleToggleStateByContainer?.[containerId]) {
            window.updateSubtitleToggleStateByContainer[containerId]();
          }
        }, 50);
      });

      subtitleItems.push(item);
      subtitleSubmenu.appendChild(item);
    });
  }

  // Hover effects are now handled by CSS
  subtitleOffItem.addEventListener("click", async (e) => {
    e.stopPropagation();
    // Use PRESTOplay TrackManager to disable subtitles
    if (player?.getTrackManager) {
      const trackManager = player.getTrackManager();
      if (trackManager?.setTextTrack) {
        try {
          await trackManager.setTextTrack(null);
        } catch (e) {
          console.error("Failed to disable subtitles:", e);
        }
      }
    }

    // Update selection highlighting
    updateSelectionHighlighting();

    // Force update subtitle item state to "Off"
    const currentSubtitle = subtitleItem.querySelector(".pp-subtitle-current");
    currentSubtitle.textContent = "Off";

    // Hide all submenus
    subtitleSubmenu.classList.add("pp-display-none");
    subtitleSubmenu.classList.remove("pp-display-block");

    updateSubtitleItemState(false);

    // Update subtitle toggle button state after a short delay
    setTimeout(() => {
      const containerId = settingsPopup.closest('.clpp-container')?.id;
      if (containerId && window.updateSubtitleToggleStateByContainer?.[containerId]) {
        window.updateSubtitleToggleStateByContainer[containerId]();
      }
    }, 50);
  });

  subtitleSubmenu.appendChild(subtitleOffItem);

  // Function to update selection highlighting based on track modes
  function updateSelectionHighlighting() {
    // Clear all selections first
    subtitleItems.forEach((item) => {
      item.classList.remove("selected");
      const checkmark = item.querySelector(".pp-subtitle-checkmark");
      if (checkmark) {
        checkmark.style.visibility = "hidden";
        checkmark.style.color = "#fff";
      }
    });

    // Check if any PRESTOplay track is active
    let hasActiveTrack = false;
    if (player?.getTrackManager) {
      const trackManager = player.getTrackManager();
      if (trackManager?.getTextTrack) {
        try {
          const activeTrack = trackManager.getTextTrack();
          if (activeTrack) {
            // Find the matching menu item
            const trackIndex = subtitleItems.findIndex(
              (track) => track.id == activeTrack.id
            );

            if (trackIndex >= 0) {
              const trackItem = subtitleItems[trackIndex];
              if (trackItem) {
                trackItem.classList.add("selected");
                const checkmark = trackItem.querySelector(".pp-subtitle-checkmark");
                if (checkmark) {
                  checkmark.style.visibility = "visible";
                  checkmark.style.color = "#2196F3";
                }
                hasActiveTrack = true;
              }
            }
          }
        } catch (e) {
          console.error("Failed to get active PRESTOplay track:", e.message);
        }
      }
    }

    if (!hasActiveTrack) {
      // No track is showing, highlight "Off"
      subtitleOffItem.classList.add("selected");
      const checkmark = subtitleOffItem.querySelector(".pp-subtitle-checkmark");
      if (checkmark) {
        checkmark.style.visibility = "visible";
        checkmark.style.color = "#2196F3";
      }
      updateSubtitleItemState(false);
    } else {
      updateSubtitleItemState(true);
    }
  }

  // Function to update subtitle item visual state
  function updateSubtitleItemState(enabled) {
    const currentSubtitle = subtitleItem.querySelector(".pp-subtitle-current");

    if (enabled) {
      // Get active track name
      const trackName = getActiveTrackName();
      currentSubtitle.textContent = trackName;
    } else {
      currentSubtitle.textContent = "Off";
    }
  }

  // Helper function to get active track name
  function getActiveTrackName() {
    try {
      if (!player?.getTrackManager) return "On";

      const trackManager = player.getTrackManager();
      if (!trackManager?.getTextTrack) return "On";

      const activeTrack = trackManager.getTextTrack();
      if (!activeTrack) return "On";

      return activeTrack.language ?? "On";
    } catch (e) {
      return "On";
    }
  }

  // Add click handler to subtitle item
  subtitleItem.addEventListener("click", (e) => {
    e.stopPropagation();
    // Hide other submenus first
    const speedSubmenu = settingsPopup.querySelector(".pp-speed-submenu");
    if (speedSubmenu) {
      speedSubmenu.classList.add("pp-display-none");
      speedSubmenu.classList.remove("pp-display-block");
    }
    const qualitySubmenu = settingsPopup.querySelector(".pp-quality-submenu");
    if (qualitySubmenu) {
      qualitySubmenu.classList.add("pp-display-none");
      qualitySubmenu.classList.remove("pp-display-block");
    }

    if (subtitleSubmenu.classList.contains("pp-display-none")) {
      subtitleSubmenu.classList.remove("pp-display-none");
      subtitleSubmenu.classList.add("pp-display-block");
    } else {
      subtitleSubmenu.classList.add("pp-display-none");
      subtitleSubmenu.classList.remove("pp-display-block");
    }
  });

  // Close submenu when clicking outside
  document.addEventListener("click", (e) => {
    if (
      !subtitleSubmenu.contains(e.target) &&
      !subtitleItem.contains(e.target)
    ) {
      subtitleSubmenu.classList.add("pp-display-none");
      subtitleSubmenu.classList.remove("pp-display-block");
    }
    // Also close other submenus
    const speedSubmenu = settingsPopup.querySelector(".pp-speed-submenu");
    if (
      speedSubmenu &&
      !speedSubmenu.contains(e.target) &&
      !settingsPopup
        .querySelector(".pp-settings-item:nth-child(1)")
        .contains(e.target)
    ) {
      speedSubmenu.classList.add("pp-display-none");
      speedSubmenu.classList.remove("pp-display-block");
    }
    const qualitySubmenu = settingsPopup.querySelector(".pp-quality-submenu");
    if (
      qualitySubmenu &&
      !qualitySubmenu.contains(e.target) &&
      !settingsPopup
        .querySelector(".pp-settings-item:nth-child(3)")
        .contains(e.target)
    ) {
      qualitySubmenu.classList.add("pp-display-none");
      qualitySubmenu.classList.remove("pp-display-block");
    }
  });

  // Initialize selection highlighting
  updateSelectionHighlighting();

  // Expose functions for external use
  window.updateSubtitleSelection = updateSelectionHighlighting;

  return {
    item: subtitleItem,
    submenu: subtitleSubmenu,
    addSubtitleTracks,
    updateSelectionHighlighting,
    updateSubtitleItemState,
  };
}