import { subtitleToggleIcon } from "../../assets/icons/index.js";

export function createSubtitleToggleComponent(player, container, isAudioOnly = false) {
  if (isAudioOnly) {
    // Return null for audio-only content
    return null;
  }

  // Create subtitle toggle wrapper
  const subtitleToggleWrapper = document.createElement("div");
  subtitleToggleWrapper.className = "pp-subtitle-toggle-wrapper";

  // Create subtitle toggle button
  const subtitleToggleBtn = document.createElement("button");
  subtitleToggleBtn.className =
    "prestoplay-control-btn pp-btn-small pp-subtitle-toggle-btn";
  subtitleToggleBtn.title = "Subtitles/CC";
  subtitleToggleBtn.innerHTML = subtitleToggleIcon;

  // Add red line indicator for active subtitle
  const subtitleIndicator = document.createElement("div");
  subtitleIndicator.className = "pp-subtitle-indicator";

  subtitleToggleWrapper.appendChild(subtitleToggleBtn);
  subtitleToggleWrapper.appendChild(subtitleIndicator);

  // Hide subtitle toggle button by default (will be shown if tracks are available)
  subtitleToggleWrapper.style.display = "none";

  // Subtitle toggle functionality
  let subtitleTracks = [];
  let isSubtitleEnabled = false;

  // Function to update subtitle toggle state
  function updateSubtitleToggleState() {
    if (player?.getTrackManager) {
      const trackManager = player.getTrackManager();
      if (trackManager?.getTextTrack) {
        try {
          const activeTrack = trackManager.getTextTrack();
          isSubtitleEnabled = activeTrack !== null;

          // Show/hide red line indicator
          if (isSubtitleEnabled) {
            subtitleIndicator.style.opacity = "1";
          } else {
            subtitleIndicator.style.opacity = "0";
          }
        } catch (e) {
          console.error("Failed to get subtitle state:", e);
          isSubtitleEnabled = false;
          subtitleIndicator.style.opacity = "0";
        }
      }
    }
  }

  // Function to toggle subtitle
  function toggleSubtitle() {
    if (player?.getTrackManager) {
      const trackManager = player.getTrackManager();
      if (trackManager?.setTextTrack) {
        try {
          // Get current state first
          const currentActiveTrack = trackManager.getTextTrack();
          const currentlyEnabled = currentActiveTrack !== null;

          if (currentlyEnabled) {
            // Turn off subtitle
            trackManager.setTextTrack(null);
            isSubtitleEnabled = false;
            subtitleIndicator.style.opacity = "0";

            // Update subtitle menu to show "Off" after a short delay
            setTimeout(() => {
              if (window.updateSubtitleSelection) {
                window.updateSubtitleSelection();
              }
            }, 50);
          } else {
            // Turn on subtitle - use first available track
            if (subtitleTracks.length > 0) {
              const firstTrack = subtitleTracks[0];
              trackManager.setTextTrack(firstTrack);
              isSubtitleEnabled = true;
              subtitleIndicator.style.opacity = "1";

              // Update subtitle menu to show first track selected after a short delay
              setTimeout(() => {
                if (window.updateSubtitleSelection) {
                  window.updateSubtitleSelection();
                }
              }, 50);
            }
          }
        } catch (e) {
          console.error("Failed to toggle subtitle:", e);
        }
      }
    }
  }

  // Add click handler
  subtitleToggleBtn.addEventListener("click", (e) => {
    e.stopPropagation();
    toggleSubtitle();
  });

  // Store subtitle tracks for later use
  const containerId = container.id;
  
  // Create container-specific functions
  window.setSubtitleTracksForToggleByContainer = window.setSubtitleTracksForToggleByContainer || {};
  window.setSubtitleTracksForToggleByContainer[containerId] = function (tracks) {
    subtitleTracks = tracks || [];

    // Hide subtitle toggle button if no tracks available
    if (subtitleTracks.length === 0) {
      subtitleToggleWrapper.style.display = "none";
    } else {
      subtitleToggleWrapper.style.display = "block";
      updateSubtitleToggleState();
    }
  };

  // Initialize with no tracks (button will be hidden)
  window.setSubtitleTracksForToggleByContainer[containerId]([]);

  // Expose update function for external calls
  window.updateSubtitleToggleStateByContainer = window.updateSubtitleToggleStateByContainer || {};
  window.updateSubtitleToggleStateByContainer[containerId] = updateSubtitleToggleState;

  return {
    wrapper: subtitleToggleWrapper,
    button: subtitleToggleBtn,
    indicator: subtitleIndicator,
    updateSubtitleToggleState,
    toggleSubtitle
  };
} 