// Timeline Component
export function createTimelineComponent(video) {
  // --- Timeline & time ---
  const timelineContainer = document.createElement("div");
  timelineContainer.className = "pp-timeline-container";
  
  // Current time
  const currentTimeEl = document.createElement("span");
  currentTimeEl.className = "pp-current-time";
  currentTimeEl.textContent = "00:00";
  
  // Progress bar
  const progressBar = document.createElement("input");
  progressBar.type = "range";
  progressBar.className = "pp-progress-bar";
  progressBar.min = 0;
  progressBar.max = 100;
  progressBar.value = 0;
  progressBar.step = 0.01;
  
  // Duration
  const durationEl = document.createElement("span");
  durationEl.className = "pp-duration";
  durationEl.textContent = "00:00";
  
  timelineContainer.appendChild(currentTimeEl);
  timelineContainer.appendChild(progressBar);
  timelineContainer.appendChild(durationEl);

  // --- Tooltip time marker ---
  let timelineTooltip;
  progressBar.addEventListener("mousemove", (e) => {
    if (!timelineTooltip) {
      timelineTooltip = document.createElement("div");
      timelineTooltip.className = "pp-timeline-tooltip";
      timelineContainer.appendChild(timelineTooltip);
    }
    const rect = progressBar.getBoundingClientRect();
    const percent = (e.clientX - rect.left) / rect.width;
    const dur = video.duration || 0;
    const hoverTime = Math.max(0, Math.min(dur, percent * dur));
    timelineTooltip.textContent = formatTime(hoverTime);
    timelineTooltip.style.left = percent * 100 + "%";
    timelineTooltip.style.opacity = "1";
  });
  
  progressBar.addEventListener("mouseleave", () => {
    if (timelineTooltip) timelineTooltip.style.opacity = "0";
  });

  // Utility function for time formatting
  function formatTime(sec) {
    if (isNaN(sec)) return "00:00";
    const m = Math.floor(sec / 60);
    const s = Math.floor(sec % 60);
    return `${m < 10 ? "0" : ""}${m}:${s < 10 ? "0" : ""}${s}`;
  }
  
  // Timeline update function
  function updateTimeline() {
    const cur = video.currentTime;
    const dur = video.duration;
    currentTimeEl.textContent = formatTime(cur);
    durationEl.textContent = formatTime(dur);
    progressBar.max = dur || 1;
    progressBar.value = cur;

    // Update progress bar background
    const progressPercent = dur > 0 ? (cur / dur) * 100 : 0;
    progressBar.style.backgroundImage = `linear-gradient(to right, #f03 0%, #f03 ${progressPercent}%, rgba(255, 255, 255, 0.3) ${progressPercent}%)`;
  }

  // Progress bar input handler
  progressBar.addEventListener("input", (e) => {
    video.currentTime = parseFloat(e.target.value);
    
    // Update progress bar background immediately
    const progressPercent = (e.target.value / e.target.max) * 100;
    progressBar.style.backgroundImage = `linear-gradient(to right, #f03 0%, #f03 ${progressPercent}%, rgba(255, 255, 255, 0.3) ${progressPercent}%)`;
  });

  // Continuous update using requestAnimationFrame
  let rafId;
  function rafUpdate() {
    updateTimeline();
    rafId = requestAnimationFrame(rafUpdate);
  }
  rafUpdate();

  // Cleanup function
  function cleanup() {
    if (rafId) {
      cancelAnimationFrame(rafId);
    }
  }

  return {
    container: timelineContainer,
    progressBar,
    currentTimeEl,
    durationEl,
    updateTimeline,
    cleanup,
    formatTime
  };
} 