// Volume Control Component
import { volumeMutedIcon, volumeLowIcon, volumeHighIcon } from "../../assets/icons/index.js";

export function createVolumeControlComponent(video) {
  const volumeWrapper = document.createElement("div");
  volumeWrapper.className = "pp-volume-wrapper";

  const volumeBtn = document.createElement("button");
  volumeBtn.className = "prestoplay-control-btn prestoplay-volume-btn";
  volumeBtn.title = "Volume";
  volumeBtn.innerHTML =
    video.volume === 0
      ? volumeMutedIcon
      : video.volume < 0.5
      ? volumeLowIcon
      : volumeHighIcon;

  // Volume slider (hidden by default, show on hover)
  const volumeSlider = document.createElement("input");
  volumeSlider.type = "range";
  volumeSlider.className = "pp-volume-slider";
  volumeSlider.min = "0";
  volumeSlider.max = "1";
  volumeSlider.step = "0.01";
  volumeSlider.value = video.volume;

  // Initialize volume slider background gradient
  const initialVolumePercent = (video.volume / 1) * 100;
  volumeSlider.style.backgroundImage = `linear-gradient(to right, #f03 0%, #f03 ${initialVolumePercent}%, rgba(255, 255, 255, 0.3) ${initialVolumePercent}%)`;

  // Function to update volume icon based on current volume and mute state
  function updateVolumeIcon() {
    const effectiveVolume = video.muted ? 0 : video.volume;
    volumeBtn.innerHTML =
      effectiveVolume === 0
        ? volumeMutedIcon
        : effectiveVolume < 0.5
        ? volumeLowIcon
        : volumeHighIcon;
  }

  // Function to handle volume change events
  function handleVolumeChange() {
    volumeSlider.value = video.volume;
    updateVolumeIcon();
    
    // Update volume slider background gradient
    const volumePercent = (video.volume / 1) * 100;
    volumeSlider.style.backgroundImage = `linear-gradient(to right, #f03 0%, #f03 ${volumePercent}%, rgba(255, 255, 255, 0.3) ${volumePercent}%)`;
  }

  // Initialize volume icon
  updateVolumeIcon();

  // Volume slider styling
  volumeSlider.addEventListener("input", (e) => {
    video.volume = parseFloat(e.target.value);
    updateVolumeIcon();
    
    // Update volume slider background gradient
    const volumePercent = (e.target.value / e.target.max) * 100;
    volumeSlider.style.backgroundImage = `linear-gradient(to right, #f03 0%, #f03 ${volumePercent}%, rgba(255, 255, 255, 0.3) ${volumePercent}%)`;
  });

  // Add click event for volume slider to set volume at clicked point
  volumeSlider.addEventListener("click", (e) => {
    e.stopPropagation();
    // Calculate the click position relative to the slider
    const rect = volumeSlider.getBoundingClientRect();
    const clickX = e.clientX - rect.left;
    const clickY = e.clientY - rect.top;
    const sliderWidth = rect.width;
    
    // Check if click is within the extended clickable area (5px above/below from CSS padding)
    // The actual slider height is 5px, but with 5px padding top and bottom, total height is 15px
    const totalHeight = 15; // 5px (slider) + 5px (padding top) + 5px (padding bottom)
    
    if (clickY < 0 || clickY > totalHeight) {
      return; // Click is outside the extended area
    }
    
    const clickPercent = clickX / sliderWidth;
    
    // Convert to volume value (0-1)
    const newVolume = Math.max(0, Math.min(1, clickPercent));
    
    // Set the volume
    video.volume = newVolume;
    volumeSlider.value = newVolume;
    updateVolumeIcon();
    
    // Update volume slider background gradient
    const volumePercent = (newVolume / 1) * 100;
    volumeSlider.style.backgroundImage = `linear-gradient(to right, #f03 0%, #f03 ${volumePercent}%, rgba(255, 255, 255, 0.3) ${volumePercent}%)`;
  });

  volumeBtn.addEventListener("click", (e) => {
    e.stopPropagation();
    video.muted = !video.muted;
    updateVolumeIcon();
  });

  // video.addEventListener("volumechange", handleVolumeChange);

  // Listen for mute state changes
  // video.addEventListener("mute", updateVolumeIcon);
  // video.addEventListener("unmute", updateVolumeIcon);

  // Show/hide volume slider on hover
  let volumeTimeout;
  volumeWrapper.addEventListener("mouseenter", () => {
    clearTimeout(volumeTimeout);
    volumeSlider.classList.add("pp-volume-slider-visible");
    volumeSlider.classList.remove("pp-volume-slider-hidden");
  });

  volumeWrapper.addEventListener("mouseleave", () => {
    volumeTimeout = setTimeout(() => {
      volumeSlider.classList.remove("pp-volume-slider-visible");
      volumeSlider.classList.add("pp-volume-slider-hidden");
    }, 300);
  });

  // Add initial classes for volume slider
  volumeSlider.classList.add("pp-volume-slider-hidden");
  volumeSlider.classList.remove("pp-volume-slider-visible");

  volumeWrapper.appendChild(volumeBtn);
  volumeWrapper.appendChild(volumeSlider);

  // Cleanup function
  function cleanup() {
    if (volumeTimeout) {
      clearTimeout(volumeTimeout);
    }
    // video.removeEventListener("volumechange", handleVolumeChange);
    // video.removeEventListener("mute", updateVolumeIcon);
    // video.removeEventListener("unmute", updateVolumeIcon);
  }

  return {
    wrapper: volumeWrapper,
    button: volumeBtn,
    slider: volumeSlider,
    updateVolumeIcon,
    cleanup
  };
} 