// Custom Controls for PRESTOplay
import { createPlayPauseButton } from "./components/play-pause/play-pause.js";
import { createTimelineComponent } from "./components/timeline/timeline.js";
import { createVolumeControlComponent } from "./components/volume-control/volume-control.js";
import { createSettingsComponent } from "./components/settings/settings.js";
import { createFullscreenComponent } from "./components/fullscreen/fullscreen.js";
import { createSubtitleToggleComponent } from "./components/subtitle-toggle/subtitle-toggle.js";
import { createTooltipComponent } from "./components/tooltip/tooltip.js";
import { createTimeDisplayComponent } from "./components/time-display/time-display.js";
import { createLoadingComponent } from "./components/loading/loading.js";

export function createCustomControls(player, video, containerId) {
  if (!player || !video || !containerId) {
    console.warn(
      "Player, video element and containerId are required for custom controls"
    );
    return;
  }

  const container = document.getElementById(containerId);

  if (!container) {
    console.error(`Container with id "${containerId}" not found`);
    return;
  }

  // Create controls container
  const controlsContainer = document.createElement("div");
  controlsContainer.className = "prestoplay-custom-controls modern";

  // --- Left controls ---
  const leftControls = document.createElement("div");
  leftControls.className = "pp-left-controls";
  // Play/Pause
  const playPauseComponent = createPlayPauseButton(video, container);
  const playPauseBtn = playPauseComponent.button;
  leftControls.appendChild(playPauseBtn);

  // --- Timeline & time ---
  const timelineComponent = createTimelineComponent(video);
  const timelineContainer = timelineComponent.container;
  const progressBar = timelineComponent.progressBar;
  const currentTimeEl = timelineComponent.currentTimeEl;
  const durationEl = timelineComponent.durationEl;

  // --- Volume & Time controls ---
  const volumeTimeContainer = document.createElement("div");
  volumeTimeContainer.className = "pp-volume-time-container";

  // Volume control component
  const volumeComponent = createVolumeControlComponent(video);
  const volumeWrapper = volumeComponent.wrapper;
  // Time display component
  const timeDisplayComponent = createTimeDisplayComponent(
    video,
    timelineComponent
  );
  const timeDisplay = timeDisplayComponent.element;

  volumeTimeContainer.appendChild(volumeWrapper);
  volumeTimeContainer.appendChild(timeDisplay);
  leftControls.appendChild(volumeTimeContainer);

  // --- Right controls ---
  const rightControls = document.createElement("div");
  rightControls.className = "pp-right-controls";

  // Check if this is audio-only content
  const isAudioOnly = container.getAttribute("data-audio-only") === "true";

  // Only add settings controls for video content (not audio-only)
  let subtitleToggleComponent;
  let settingsComponent;
  let fullscreenComponent;
  if (!isAudioOnly) {
    // Subtitle toggle component
    subtitleToggleComponent = createSubtitleToggleComponent(
      player,
      container,
      isAudioOnly
    );
    const subtitleToggleWrapper = subtitleToggleComponent.wrapper;
    rightControls.appendChild(subtitleToggleWrapper);

    // Settings component
    settingsComponent = createSettingsComponent(video, player, container);
    const settingsWrapper = settingsComponent.wrapper;
    rightControls.appendChild(settingsWrapper);

    // Fullscreen component
    fullscreenComponent = createFullscreenComponent(
      container,
      video,
      isAudioOnly
    );
    const fullscreenBtn = fullscreenComponent.button;
    rightControls.appendChild(fullscreenBtn);
  }

  // --- Assemble control bar ---
  controlsContainer.appendChild(timelineContainer);
  const controlsRow = document.createElement("div");
  controlsRow.className = "pp-controls-row";
  controlsRow.appendChild(leftControls);
  controlsRow.appendChild(rightControls);
  controlsContainer.appendChild(controlsRow);
  container.appendChild(controlsContainer);

  // --- Loading component ---
  const loadingComponent = createLoadingComponent(video, container);

  // --- Tooltip logic ---
  const tooltipComponent = createTooltipComponent();
  const tooltipButtons = [playPauseBtn];

  // Add volume control button to tooltips if it exists
  if (volumeComponent) {
    tooltipButtons.push(volumeComponent.button);
  }
  // Add subtitle toggle button to tooltips if it exists
  if (subtitleToggleComponent) {
    tooltipButtons.push(subtitleToggleComponent.button);
  }
  // Add settings button to tooltips if it exists
  if (settingsComponent) {
    tooltipButtons.push(settingsComponent.button);
  }
  // Only add fullscreen to tooltips for video content (not audio-only)
  if (!isAudioOnly && fullscreenComponent) {
    tooltipButtons.push(fullscreenComponent.button);
  }

  // Setup tooltips for all buttons
  tooltipComponent.setupTooltips(tooltipButtons);

  // --- Control visibility logic ---
  function showControlBar() {
    controlsContainer.style.opacity = "1";
    controlsContainer.style.pointerEvents = "auto";
    container.classList.remove("hide-cursor");

    // Show title overlay when controls are shown
    if (window.showTitleOverlay) {
      window.showTitleOverlay();
    }
  }

  function hideControlBar() {
    // Don't hide controls if this is audio-only content
    if (container.getAttribute("data-audio-only") === "true") {
      return;
    }

    // Only hide controls in fullscreen mode
    if (document.fullscreenElement || 
        document.webkitFullscreenElement || 
        document.mozFullScreenElement || 
        document.msFullscreenElement) {
      controlsContainer.style.opacity = "0";
      controlsContainer.style.pointerEvents = "none";
      container.classList.add("hide-cursor");

      // Hide title overlay when controls are hidden
      if (window.hideTitleOverlay) {
        window.hideTitleOverlay();
      }

      // Hide settings popup when control bar is hidden
      if (settingsComponent && settingsComponent.hideAllSubmenus) {
        settingsComponent.hideAllSubmenus();
      }
      if (settingsComponent && settingsComponent.hidePopup) {
        settingsComponent.hidePopup();
      }
    }
  }

  // Set control bar functions for fullscreen component
  if (fullscreenComponent) {
    fullscreenComponent.setControlBarFunctions(showControlBar, hideControlBar);
  }

  // Show controls on mouse enter
  container.addEventListener("mouseenter", showControlBar);

  // Hide controls on mouse leave
  container.addEventListener("mouseleave", hideControlBar);

  // Keep controls visible when hovering over them
  controlsContainer.addEventListener("mouseenter", showControlBar);
  controlsContainer.addEventListener("mouseleave", (e) => {
    // Only hide if mouse is not over the container and not audio-only
    if (
      !container.contains(e.relatedTarget) &&
      container.getAttribute("data-audio-only") !== "true"
    ) {
      hideControlBar();
    }
  });

  // Initially show controls
  showControlBar();

  const returnObject = {
    controlsContainer,
    playPauseBtn,
    volumeComponent,
    volumeControl: volumeWrapper,
    timelineComponent,
    settingsComponent,
    progressBar,
    currentTimeEl,
    durationEl,
    timeDisplay,
    timeDisplayComponent,
    loadingComponent,
  };

  // Only include fullscreen button for video content (not audio-only)
  if (!isAudioOnly && fullscreenComponent) {
    returnObject.fullscreenBtn = fullscreenComponent.button;
    returnObject.fullscreenComponent = fullscreenComponent;
    returnObject.subtitleToggleBtn = subtitleToggleComponent?.button;
    returnObject.subtitleToggleComponent = subtitleToggleComponent;
  }

  // Store controls with container-specific key
  window.prestoplayControlsByContainer = window.prestoplayControlsByContainer || {};
  window.prestoplayControlsByContainer[containerId] = returnObject;

  return returnObject;
}

// Add subtitle selection to settings popup if subtitle tracks exist
window.addSubtitleButtonToControls = function (containerId, prestoTracks) {
  const container = document.getElementById(containerId);

  if (!container) {
    return;
  }

  // Skip subtitle controls for audio-only content
  if (container?.getAttribute("data-audio-only") === "true") {
    return;
  }

  // Get settings component from container-specific storage
  const controls = window.prestoplayControlsByContainer?.[containerId];

  if (!controls || !controls.settingsComponent) {
    return;
  }

  const settingsComponent = controls.settingsComponent;
  const subtitleComponent = settingsComponent.subtitleComponent;

  // Always set subtitle tracks for toggle button (even if empty)
  if (window.setSubtitleTracksForToggleByContainer?.[containerId]) {
    window.setSubtitleTracksForToggleByContainer[containerId](prestoTracks || []);
  }

  // If no subtitle tracks, don't add subtitle submenu
  if (!prestoTracks || prestoTracks.length === 0) {
    return;
  }

  // Add subtitle tracks to subtitle component
  subtitleComponent.addSubtitleTracks(prestoTracks);
};

// Add quality selection to settings popup
window.addQualityButtonToControls = function (containerId, videoTracks) {
  const container = document.getElementById(containerId);

  // Skip quality controls for audio-only content
  if (container?.getAttribute("data-audio-only") === "true") {
    return;
  }

  if (!container) {
    return;
  }

  // Get settings component from container-specific storage
  const controls = window.prestoplayControlsByContainer?.[containerId];
  if (!controls || !controls.settingsComponent) {
    return;
  }

  const settingsComponent = controls.settingsComponent;
  const qualityComponent = settingsComponent.qualityComponent;

  // Add video tracks to quality component
  qualityComponent.addVideoTracks(videoTracks);
};
