// DRM Configuration for DRMtoday integration with PRESTOplay

export const DRM_TYPES = {
  WIDEVINE: 'com.widevine.alpha',
  PLAYREADY: 'com.microsoft.playready',
  FAIRPLAY: 'com.apple.fps.1_0'
};

export function setupDRM(player, drmConfig = {}) {
  if (!player || !drmConfig) {
    console.warn('DRM setup requires player and drmConfig');
    return;
  }

  // Default DRMtoday configuration
  const defaultDRMConfig = {
    env: 'DRMtoday',
    customData: {
      userId: 'default-user',
      sessionId: 'default-session',
      merchant: 'default-merchant'
    }
  };

  // Merge with user configuration
  const finalDRMConfig = {
    ...defaultDRMConfig,
    ...drmConfig
  };

  // Setup DRM for PRESTOplay
  setupPrestoPlayDRM(player, finalDRMConfig);
}

function setupPrestoPlayDRM(player, drmConfig) {
  if (!player || !player.load) {
    console.error('Invalid PRESTOplay Player instance');
    return;
  }

  // Add DRM event listeners
  addDRMEventListeners(player);
}

function addDRMEventListeners(player) {
  player.addEventListener('drm-session-update', (event) => {
    console.log('DRM Session Update:', event);
  });

  player.addEventListener('drm-session-removed', (event) => {
    console.log('DRM Session Removed:', event);
  });

  player.addEventListener('drm-error', (event) => {
    console.error('DRM Error:', event);
    handleDRMError(event, player);
  });
}

function handleDRMError(error, player) {
  console.error('DRM Error Details:', error);
  
  // Emit custom DRM error event
  const customEvent = new CustomEvent('prestoplay-drm-player:drm-error', {
    detail: {
      error: error,
      player: player
    }
  });
  document.dispatchEvent(customEvent);
  
  // Display user-friendly error message
  const errorElement = document.createElement('div');
  errorElement.className = 'prestoplay-drm-error';
  errorElement.textContent = 'Cannot play DRM-protected video. Please check your DRM configuration.';
  errorElement.style.cssText = `
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    background: rgba(0, 0, 0, 0.8);
    color: white;
    padding: 20px;
    border-radius: 5px;
    z-index: 1000;
  `;
  
  const videoElement = player.video;
  if (videoElement && videoElement.parentElement) {
    videoElement.parentElement.appendChild(errorElement);
  }
}

// Helper function to detect DRM support
export function detectDRMSupport() {
  const support = {
    widevine: false,
    playready: false,
    fairplay: false
  };

  if (window.navigator && window.navigator.requestMediaKeySystemAccess) {
    // Check Widevine support
    navigator.requestMediaKeySystemAccess(DRM_TYPES.WIDEVINE, [{
      initDataTypes: ['cenc'],
      videoCapabilities: [{
        contentType: 'video/mp4; codecs="avc1.42E01E"'
      }]
    }]).then(() => {
      support.widevine = true;
    }).catch(() => {
      support.widevine = false;
    });

    // Check PlayReady support
    navigator.requestMediaKeySystemAccess(DRM_TYPES.PLAYREADY, [{
      initDataTypes: ['cenc'],
      videoCapabilities: [{
        contentType: 'video/mp4; codecs="avc1.42E01E"'
      }]
    }]).then(() => {
      support.playready = true;
    }).catch(() => {
      support.playready = false;
    });

    // Check FairPlay support (Safari only)
    if (window.WebKitMediaKeys) {
      support.fairplay = true;
    }
  }

  return support;
}