import { clpp } from "@castlabs/prestoplay";
import "@castlabs/prestoplay/cl.core";
import "@castlabs/prestoplay/cl.mse";
import "@castlabs/prestoplay/cl.dash";
import "@castlabs/prestoplay/cl.hls";
import "@castlabs/prestoplay/cl.smooth";
import "@castlabs/prestoplay/cl.vtt";
import "@castlabs/prestoplay/cl.ttml";
import "@castlabs/prestoplay/cl.htmlcue";
import "@castlabs/prestoplay/clpp.styles.css";
import "./styles/index.css";
import { createCustomControls } from "./custom-controls";
import { checkAudioOnlyContent } from "./audio-only-handler";
import { setupSubtitleDetection } from "./subtitle-handler";
import { setupTitleDetection, cleanupTitleOverlay } from "./title-handler";

class PrestoPlayDRMPlayer {
  constructor(options = {}) {
    if (!options.containerId) throw new Error("containerId is required");
    if (!options.source) throw new Error("source is required");
    if (!options.prestoplayCred?.license && !options.prestoplayCred?.viewerId)
      throw new Error(
        "license and viewerId is required to initilize a PRESTOplay player"
      );

    this.options = {
      width: undefined,
      height: undefined,
      autoplay: false,
      loop: false,
      muted: false,
      controls: false,
      prestoplayCred: {
        license: undefined,
        viewerId: undefined,
      },
      isDRM: false,
      drmConfig: undefined,
      ...options,
    };

    this.container = document.getElementById(this.options.containerId);
    if (!this.container)
      throw new Error(
        `Container with id "${this.options.containerId}" not found`
      );

    this.init();
  }

  async init() {
    this.container.innerHTML = "";
    this.container.className = "clpp-container";

    // Reset video container hidden state for new video
    this.videoContainerHidden = false;

    // Reset container styles for new video (clear audio-only styling)
    this.container.style.background = "";
    this.container.style.border = "";

    // Reset removed controls tracking
    this.removedControlsForAudio = false;

    // Make this instance globally accessible for subtitle controls
    window.prestoPlayerInstance = this;

    this.container.style.boxSizing = "content-box";
    this.container.style.borderRadius = "12px";

    this.video = document.createElement("video");
    this.video.style.width = "100%";
    this.video.style.height = "100%";
    this.video.style.display = "block";
    this.video.style.background = "#000";
    
    // Ensure no controls attribute is set (prevents iOS Safari built-in controls)
    this.video.removeAttribute("controls");
    this.video.setAttribute("playsinline", "");
    this.video.setAttribute("webkit-playsinline", "");
    this.video.setAttribute("x-webkit-airplay", "allow");

    // Wrapper for video - TTML creates its own container
    const videoWrapper = document.createElement("div");
    videoWrapper.style.cssText = `
      position: relative;
      width: 100%;
      height: 100%;
    `;

    videoWrapper.appendChild(this.video);
    this.container.appendChild(videoWrapper);
    
    // Set container to flex layout for video + controls
    this.container.style.display = "flex";
    this.container.style.flexDirection = "column";
    this.container.style.alignItems = "stretch";

    console.log("Support HLS", clpp.hls);
    clpp.install(clpp.dash.DashComponent);
    clpp.install(clpp.hls.HlsComponent);
    if (clpp.smooth && clpp.smooth.SmoothComponent) {
      clpp.install(clpp.smooth.SmoothComponent);
    }

    // Always install subtitle components initially - they will be disabled later if TTML parsing succeeds
    clpp.install(clpp.vtt.VttComponent);
    clpp.install(clpp.ttml.TtmlComponent);
    clpp.install(clpp.htmlcue.HtmlCueComponent);

    this.player = new clpp.Player(this.video, this.options.prestoplayCred);

    this.player.use(clpp.dash.DashComponent);
    this.player.use(clpp.hls.HlsComponent);
    if (clpp.smooth && clpp.smooth.SmoothComponent) {
      this.player.use(clpp.smooth.SmoothComponent);
    }

    // Always enable subtitle components initially - they will be disabled later if TTML parsing succeeds
    if (clpp.vtt && clpp.vtt.VttComponent) {
      this.player.use(clpp.vtt.VttComponent);
    }
    if (clpp.ttml && clpp.ttml.TtmlComponent) {
      this.player.use(clpp.ttml.TtmlComponent);
    }
    if (clpp.htmlcue && clpp.htmlcue.HtmlCueComponent) {
      this.player.use(clpp.htmlcue.HtmlCueComponent);
    }

    try {
      // Load video
      const options = {
        source: this.options.source,
        autoplay: this.options.autoplay,
        loop: this.options.loop,
        muted: this.options.muted,
      };

      if (this.options.isDRM && this.options.drmConfig) {
        options.drm = this.options.drmConfig;
      }

      await this.player.load(options);

      // Update container height based on video aspect ratio if height was not provided
      if (!this.options.height) {
        this.updateContainerHeight();
      }

      // Check if content is audio-only and adjust UI accordingly
      checkAudioOnlyContent(this);

      // First create custom controls
      createCustomControls(this.player, this.video, this.options.containerId);

      // Then setup subtitle detection after controls are ready
      setupSubtitleDetection(this);

      // Setup title detection after subtitles
      setupTitleDetection(this);

      // Setup quality selector after subtitles
      this.setupQualitySelector();
    } catch (err) {
      console.error("PRESTOplay load error:", err);

      // First check for audio-only content even if video load fails
      checkAudioOnlyContent(this);

      // Still try to initialize controls even if video load fails
      createCustomControls(this.player, this.video, this.options.containerId);

      // And still try subtitle detection
      setupSubtitleDetection(this);

      // And still try title detection
      setupTitleDetection(this);

      // And still try quality selector
      this.setupQualitySelector();
    }
  }

  // Quality selector
  setupQualitySelector() {
    // Skip quality selector for audio-only content
    if (this.container?.getAttribute("data-audio-only") === "true") {
      console.log("Skipping quality selector for audio-only content");
      return;
    }

    // Check PRESTOplay video track APIs immediately
    this.detectVideoTracks("immediate");
  }

  detectVideoTracks(source) {
    try {
      const trackManager = this.player.getTrackManager();

      // Use getVideoTracks method (returns all video tracks)
      if (trackManager.getVideoTracks) {
        const videoTracks = trackManager.getVideoTracks();

        if (videoTracks && videoTracks?.length > 0) {
          // Add quality selector to UI
          this.addQualitySelector(videoTracks);
          return;
        } else {
          console.log("No video tracks found in TrackManager");
        }
      }
    } catch (e) {
      console.log("TrackManager video tracks access failed:", e.message);
    }
  }

  addQualitySelector(videoTracks) {
    // Add quality selector to UI
    if (typeof window.addQualityButtonToControls === "function") {
      window.addQualityButtonToControls(this.options.containerId, videoTracks);
    } else {
      console.log("addQualityButtonToControls function not found");
    }
  }

  updateContainerHeight(videoWidth) {
    const updateHeight = () => {
      let videoWidth = this.video.videoWidth || 0;
      let videoHeight = this.video.videoHeight || 0;
      
      if (videoWidth && videoHeight) {
        // Calculate the actual width of the video element
        const containerWidth = this.container.offsetWidth || this.options.width || 800;
        const calculatedVideoHeight = containerWidth * (videoHeight / videoWidth);
        
        // Control bar height (timeline + controls row + padding)
        const controlBarHeight = 75; // Approximate height of control bar
        
        // Total height = video height + control bar height
        const totalHeight = calculatedVideoHeight + controlBarHeight;
        
        // Update container height to include control bar
        this.container.style.height = totalHeight + "px";

        this.video.removeEventListener("canplay", updateHeight);
        this.video.removeEventListener("loadedmetadata", updateHeight);
      }
    };

    // Listen to both canplay and loadedmetadata events
    this.video.addEventListener("canplay", updateHeight);
    this.video.addEventListener("loadedmetadata", updateHeight);
    
    // Also listen for resize events to recalculate height
    this.handleResize = () => {
      if (!this.options.height) {
        updateHeight();
      }
    };
    
    window.addEventListener('resize', this.handleResize);
  }

  play() {
    this.player && this.player.play();
  }
  pause() {
    this.player && this.player.pause();
  }
  stop() {
    this.player && this.player.pause();
    this.player && this.player.seekTo(0);
  }
  dispose() {
    // Clean up resize event listener
    if (this.handleResize) {
      window.removeEventListener('resize', this.handleResize);
      this.handleResize = null;
    }

    // Clean up TTML event listeners
    if (this.ttmlEventListeners) {
      this.ttmlEventListeners.forEach((listener) => {
        // Remove from all possible events
        this.video.removeEventListener("timeupdate", listener);
        this.video.removeEventListener("seeked", listener);
        this.video.removeEventListener("loadeddata", listener);
      });
      this.ttmlEventListeners = [];
    }

    // Clean up TTML container
    const ttmlContainer = document.querySelector(".ttml-subtitle-container");
    if (ttmlContainer && ttmlContainer.parentNode) {
      ttmlContainer.parentNode.removeChild(ttmlContainer);
    }

    // Clean up title overlay
    cleanupTitleOverlay(this.container);

    // Clean up loading component
    const controls = window.prestoplayControlsByContainer?.[this.options.containerId];
    if (controls && controls.loadingComponent) {
      controls.loadingComponent.cleanup();
    }

    if (this.player) {
      this.player.destroy();
      this.player = null;
    }
    if (this.video && this.video.parentNode)
      this.video.parentNode.removeChild(this.video);
  }
}

export default PrestoPlayDRMPlayer;
