// Title Handler for PRESTOplay
import shareIcon from "./assets/icons/share.svg";
import checkIcon from "./assets/icons/check.svg";

export function setupTitleDetection(playerInstance) {
  if (!playerInstance || !playerInstance.player || !playerInstance.container) {
    console.warn(
      "Player instance, player, and container are required for title detection"
    );
    return;
  }

  const { player, container, video } = playerInstance;

  // Try to get title from manifest using different approaches
  const extractTitle = async () => {
    let title = playerInstance.options.title ?? null;

    try {
      // Method 1: Try to get from PRESTOplay manifest metadata (most reliable)
      if (!title && player.getManifest) {
        const manifest = player.getManifest();
        if (manifest && manifest.title) {
          title = manifest.title;
        } else if (manifest && manifest.metadata && manifest.metadata.title) {
          title = manifest.metadata.title;
        }
      }

      // Method 2: Fallback - fetch manifest manually and parse for title
      if (!title) {
        title = await fetchManifestTitle(playerInstance.options.source);
      }
    } catch (error) {
      console.error("Title extraction failed:", error.message);
    }

    // If we found a title, create the overlay
    if (title) {
      createTitleOverlay(title, container);
    }
  };

  // Try to extract title after a short delay to ensure player is ready
  setTimeout(extractTitle, 500);

  // Also try when PRESTOplay manifest is loaded (most reliable event)
  if (player && player.getEventManager) {
    const eventManager = player.getEventManager();
    if (eventManager) {
      eventManager.addEventListener("manifestloaded", extractTitle);
    }
  }
}

// Fetch manifest manually and parse for title
async function fetchManifestTitle(sourceUrl) {
  try {
    const response = await fetch(sourceUrl);
    const manifestText = await response.text();

    // Parse DASH manifest for title
    if (sourceUrl.includes(".mpd")) {
      return parseDASHManifestTitle(manifestText);
    }

    // Parse HLS manifest for title
    if (sourceUrl.includes(".m3u8")) {
      return parseHLSManifestTitle(manifestText);
    }

    // Parse Smooth Streaming manifest for title
    if (sourceUrl.includes(".ism")) {
      return parseSmoothManifestTitle(manifestText);
    }
  } catch (error) {
    console.error("Failed to fetch manifest:", error.message);
  }

  return null;
}

// Parse DASH manifest for title
function parseDASHManifestTitle(manifestText) {
  try {
    // Look for title in ProgramInformation
    const programInfoMatch = manifestText.match(
      /<ProgramInformation[^>]*>[\s\S]*?<Title[^>]*>([^<]+)<\/Title>/i
    );
    if (programInfoMatch) {
      return programInfoMatch[1].trim();
    }

    // Look for title in AdaptationSet
    const adaptationSetMatch = manifestText.match(
      /<AdaptationSet[^>]*>[\s\S]*?<Title[^>]*>([^<]+)<\/Title>/i
    );
    if (adaptationSetMatch) {
      return adaptationSetMatch[1].trim();
    }

    // Look for title in Period
    const periodMatch = manifestText.match(
      /<Period[^>]*>[\s\S]*?<Title[^>]*>([^<]+)<\/Title>/i
    );
    if (periodMatch) {
      return periodMatch[1].trim();
    }
  } catch (error) {
    console.error("DASH manifest parsing failed:", error.message);
  }

  return null;
}

// Parse HLS manifest for title
function parseHLSManifestTitle(manifestText) {
  try {
    // Look for title in EXTINF tags
    const extinfMatch = manifestText.match(/#EXTINF:[^,]*,\s*([^\n]+)/i);
    if (extinfMatch) {
      return extinfMatch[1].trim();
    }

    // Look for title in custom tags
    const titleMatch = manifestText.match(/#EXT-X-TITLE[^:]*:\s*([^\n]+)/i);
    if (titleMatch) {
      return titleMatch[1].trim();
    }
  } catch (error) {
    console.error("HLS manifest parsing failed:", error.message);
  }

  return null;
}

// Parse Smooth Streaming manifest for title
function parseSmoothManifestTitle(manifestText) {
  try {
    // Look for title in Smooth Streaming manifest
    const titleMatch = manifestText.match(/<Title[^>]*>([^<]+)<\/Title>/i);
    if (titleMatch) {
      return titleMatch[1].trim();
    }
  } catch (error) {
    console.error("Smooth manifest parsing failed:", error.message);
  }

  return null;
}

// Create title overlay
function createTitleOverlay(title, container) {
  if (!title || !container) {
    return;
  }

  // Remove existing title overlay if any
  const existingTitle = container.querySelector(".pp-title-overlay");
  if (existingTitle) {
    existingTitle.remove();
  }

  // Create title overlay container
  const titleOverlay = document.createElement("div");
  titleOverlay.className = "pp-title-overlay";
  titleOverlay.style.cssText = `
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    background: linear-gradient(180deg, rgba(0,0,0,0.8) 0%, rgba(0,0,0,0.4) 50%, transparent 100%);
    color: white;
    padding: 16px 20px;
    font-size: 18px;
    font-weight: 600;
    line-height: 1.4;
    z-index: 5;
    opacity: 0;
    transform: translateY(-10px);
    transition: opacity 0.3s ease, transform 0.3s ease;
    pointer-events: auto;
    text-shadow: 0 1px 3px rgba(0,0,0,0.8);
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 12px;
  `;

  // Create title text
  const titleText = document.createElement("div");
  titleText.textContent = title;
  titleText.style.cssText = `
    flex: 1;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
    min-width: 0;
  `;

  // Create share button (YouTube style)
  const shareButton = document.createElement("button");
  shareButton.className = "pp-share-button";
  shareButton.title = "Share";
  shareButton.innerHTML = `
    ${shareIcon}
    <span>Share</span>
  `;
  shareButton.style.cssText = `
    background: transparent;
    border: none;
    color: white;
    padding: 8px 12px;
    border-radius: 18px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 500;
    min-width: auto;
    height: auto;
    display: flex;
    align-items: center;
    justify-content: center;
    gap: 6px;
    transition: all 0.2s ease;
    flex-shrink: 0;
  `;

  // Share button click handler
  shareButton.addEventListener("click", async (e) => {
    e.stopPropagation();

    try {
      // Get current video URL
      const videoUrl = window.location.href;

      // Copy to clipboard
      await navigator.clipboard.writeText(videoUrl);

      // Show success feedback
      const originalHTML = shareButton.innerHTML;
      shareButton.innerHTML = `
        ${checkIcon}
        <span>Copied!</span>
      `;

      // Reset after 2 seconds
      setTimeout(() => {
        shareButton.innerHTML = originalHTML;
        shareButton.style.background = "transparent";
        shareButton.style.color = "white";
        shareButton.title = "Share";
      }, 2000);
    } catch (error) {
      console.error("Failed to copy link:", error);

      // Fallback for older browsers
      const textArea = document.createElement("textarea");
      textArea.value = window.location.href;
      document.body.appendChild(textArea);
      textArea.select();
      document.execCommand("copy");
      document.body.removeChild(textArea);

      // Show feedback
      shareButton.style.background = "rgba(76, 175, 80, 0.8)";
      shareButton.style.color = "white";
      shareButton.title = "Link copied!";

      setTimeout(() => {
        shareButton.style.background = "transparent";
        shareButton.style.color = "white";
        shareButton.title = "Share";
      }, 2000);
    }
  });

  // Assemble title overlay
  titleOverlay.appendChild(titleText);
  titleOverlay.appendChild(shareButton);

  container.appendChild(titleOverlay);

  // Sync title overlay with control bar behavior
  function showTitle() {
    // Don't show title for audio-only content
    if (container.getAttribute("data-audio-only") === "true") {
      return;
    }
    titleOverlay.style.opacity = "1";
    titleOverlay.style.transform = "translateY(0)";
  }

  function hideTitle() {
    // Don't hide title for audio-only content
    if (container.getAttribute("data-audio-only") === "true") {
      return;
    }
    titleOverlay.style.opacity = "0";
    titleOverlay.style.transform = "translateY(-10px)";
  }

  // Expose functions for control bar to call
  window.showTitleOverlay = showTitle;
  window.hideTitleOverlay = hideTitle;

  // Initially show title (same as control bar)
  showTitle();
}

// Clean up title overlay
export function cleanupTitleOverlay(container) {
  if (container) {
    const titleOverlay = container.querySelector(".pp-title-overlay");
    if (titleOverlay) {
      titleOverlay.remove();
    }
    // Clean up window functions
    if (window.showTitleOverlay) {
      delete window.showTitleOverlay;
    }
    if (window.hideTitleOverlay) {
      delete window.hideTitleOverlay;
    }
  }
}
