import Session from '../sessions/Session';
/**
 * NpawProductAnalytics class is the base of npaw productAnalytics.
 * Every plugin will have an instance.
 */
export default class ProductAnalytics {
    private static VIDEOKEY_DEFAULT_IDENTIFIER;
    private initialized;
    private session;
    private settings;
    private page;
    private config;
    private searchQuery;
    private url;
    private contentHighlighted;
    private contentHighlightTimeout;
    private pendingVideoEvents;
    private adapters;
    private getReferral;
    private setAnalyticsOptions;
    private addVideoRequestListener;
    private removeVideoRequestListener;
    private videoListenerAccessor;
    private listeners;
    /**
     * Constructs the NpawProductAnalytics class
     */
    constructor(session: Session, getReferral: () => string, setAnalyticsOptions: (options: any) => void, addVideoRequestListener: (listener: any) => void, removeVideoRequestListener: (listener: any) => void);
    /**
     * Initializes product analytics
     * @param {Object} productAnalyticsSettings product analytics settings
     * @param {Object} [npawtmConfigValue] configuration settings
     */
    initialize(productAnalyticsSettings: any): void;
    /**
     * Identify current page
     * @private
     */
    private identify;
    /**
     * Destroy
     */
    destroy(): void;
    /**
     * Track adapter start
     * @private
     */
    private adapterTrackStart;
    /**
     * Execute after adapter is set to plugin
     * @param {object} adapter Video adapter
     * @param {string} [videoKey] Custom video identifier
     */
    registerAdapter(adapter: any, videoKey?: string): void;
    /**
     * Execute before removing adapter from plugin
     * @param {string} [videoKey] Custom video identifier
     */
    removeAdapter(videoKey?: string): void;
    /**
     * Get adapter bound to video key
     * @param videoKey
     * @returns
     */
    private getAdapter;
    /**
     * Get video key
     * @param videoKey
     * @returns
     */
    private getVideoKey;
    /**
     * Video listener
     * @param serviceName
     * @param videoKey
     * @param params
     */
    private videoListener;
    /**
     * New user session
     */
    newSession(onSuccess?: () => void, onFail?: () => void): void;
    /**
     * Ends user session
     */
    endSession(onSuccess?: () => void, onFail?: () => void): void;
    /**
     * Successful login
     * @param {string} userId The unique identifier of the user
     * @param {string} [profileId] The unique identifier of the profile
     * @param {string} [profileType] The profile type
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    loginSuccessful(userId: string | number, profileId: string, profileType: string, dimensions: object, metrics: object, onSuccess?: () => void, onFail?: () => void): void;
    /**
     * Login unsuccessful
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    loginUnsuccessful(dimensions?: object, metrics?: object): void;
    /**
     * Logout
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    logout(dimensions?: object, metrics?: object, onSuccess?: () => void, onFail?: () => void): void;
    /**
     * User profile created
     * @param {string} profileId The unique identifier of the profile
     * @param {string} [profileType] The profile type
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    userProfileCreated(profileId: string, profileType?: string, dimensions?: object, metrics?: object): void;
    /**
     * User profile selected
     * @param {string} profileId The unique identifier of the profile
     * @param {string} [profileType] The profile type
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    userProfileSelected(profileId: string, profileType?: string, dimensions?: object, metrics?: object, onSuccess?: () => void, onFail?: () => void): void;
    /**
     * Fire user profile selected event
     * @param {string} profileId The unique identifier of the profile
     * @param {string} [profileType] The profile type
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     * @private
     */
    _userProfileSelectedEvent(profileId: string, profileType?: string, dimensions?: object, metrics?: object): void;
    /**
     * User profile deleted
     * @param {string} profileId The unique identifier of the profile
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    userProfileDeleted(profileId: string, dimensions?: object, metrics?: object): void;
    /**
     * Get user profile dimensions
     * @param {string} profileId
     * @param {string} profileType
     * @returns
     * @private
     */
    _getUserProfileDimensions(profileId: string, profileType?: string | null): {
        profileId: string;
    };
    /**
     * Tracks navigation by route
     * @param {string} pageName The unique name to identify a page of the application.
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    trackNavByRoute(route: string, dimensions: object, metrics: object): void;
    /**
     * Tracks navigation
     * @param {string} pageName The unique name to identify a page of the application.
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    trackNavByName(pageName: string, dimensions?: object, metrics?: object): void;
    /**
     * Automatically tracks navigation when page is cached by browser
     * @param {Object} event
     * @private
     */
    private trackNavigationCached;
    /**
     * Tracks navigation (either manually or automatically)
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     * @private
     */
    private trackNavigationInternal;
    /**
     * Tracks attribution
     * @param {string} utmSource The UTM Source parameter. It is commonly used to identify a search engine, newsletter, or other source (i.e., Google, Facebook, etc.).
     * @param {string} utmMedium The UTM Medium parameter. It is commonly used to identify a medium such as email or cost-per-click (cpc).
     * @param {string} utmCampaign The UTM Campaign parameter. It is commonly used for campaign analysis to identify a specific product promotion or strategic campaign (i.e., spring sale).
     * @param {string} utmTerm The UTM Term parameter. It is commonly used with paid search to supply the keywords for ads (i.e., Customer, NonBuyer, etc.).
     * @param {string} utmContent The UTM Content parameter. It is commonly used for A/B testing and content-targeted ads to differentiate ads or links that point to the same URL (i.e., Banner1, Banner2, etc.)
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    trackAttribution(utmSource: string, utmMedium?: string, utmCampaign?: string, utmTerm?: string, utmContent?: string, dimensions?: object, metrics?: object): void;
    /**
     * Tracks attribution (either manually or automatically)
     * @param {URLSearchParams} params Object where to look for UTM params.
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     * @private
     */
    private trackAttributionInternal;
    /**
     * Retrieves UTM params from querystring (utm_source, utm_medium, utm_campaign, utm_term, utm_content) and returns an object containing them but
     * with key formatted as specified by utmFormatDot.
     * @param {URLSearchParams} params Object where to look for UTM params
     * @returns {{}}
     * @private
     */
    _getUTMParams(params: URLSearchParams): object | null;
    /**
     * Section goes into viewport.
     * @param {string} section The section title. It is commonly used to indicate the section title presented over a grid layout (e.g. Recommended Movies, Continue Watching, etc).
     * @param {Number} sectionOrder The section order within the page.
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    trackSectionVisible(section: string, sectionOrder: number, dimensions?: object, metrics?: object): void;
    /**
     * Section goes out of viewport.
     * @param {string} section The section title. It is commonly used to indicate the section title presented over a grid layout (e.g. Recommended Movies, Continue Watching, etc).
     * @param {Number} sectionOrder The section order within the page.
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    trackSectionHidden(section: string, sectionOrder: number, dimensions?: object, metrics?: object): void;
    /**
     * Sends a content highlight event if content is focused during, at least, highlightContentAfter ms.
     * @param {string} section The section title. It is commonly used to indicate the section title presented over a grid layout (e.g. Recommended Movies, Continue Watching, etc).
     * @param {Number} sectionOrder The section order within the page.
     * @param {integer} column Used to indicate the column number where content is placed in a grid layout The first column is number 1.
     * @param {integer} row Used to indicate the row number where content is placed in a grid layout. The first row is number 1. In the case of a horizontal list instead of a grid, the row parameter should be set to 1.
     * @param {string} contentID The unique content identifier of the content linked.
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    contentFocusIn(section: string, sectionOrder: number, column: number, row: number, contentID: string, dimensions?: object, metrics?: object): void;
    /**
     * Content loses focus
     * @private
     */
    contentFocusOut(): void;
    /**
     * Sends a content highlight event using selected content info
     * @private
     */
    _trackContentHiglight(): void;
    /**
     * Tracks the location of user clicks.
     * @param {string} section The section title. It is commonly used to indicate the section title presented over a grid layout (e.g. Recommended Movies, Continue Watching, etc).
     * @param {Number} sectionOrder The section order within the page.
     * @param {integer} column Used to indicate the column number where content is placed in a grid layout The first column is number 1.
     * @param {integer} row Used to indicate the row number where content is placed in a grid layout. The first row is number 1. In the case of a horizontal list instead of a grid, the row parameter should be set to 1.
     * @param {string} contentID The unique content identifier of the content linked.
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    trackContentClick(section: string, sectionOrder: number, column: number, row: number, contentID: string, dimensions?: object, metrics?: object): void;
    /**
     * Tracks when a content starts playing be it automatically or through a user interaction.
     * @param {string} contentID The unique content identifier of the content being played.
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     * @param {string} [videoKey] Custom video identifier
     * @note Apparently, VideoAnalyticsRequestHandler queues requests until all mandatory events
     * (INIT / START) are sent. Then, it sends all the queued events. This is why there is no need
     * of queueing events here.
     */
    trackPlay(contentID: string, dimensions?: object, metrics?: object, videoKey?: string): void;
    /**
     * Tracks content watching events.
     * TODO: add (2nd) argument to tell whether user state must be updated or not
     * @param {string} eventName The name of the interaction (i.e., Pause, Seek, Skip Intro, Skip Ads, Switch Language, etc.).
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     * @param {string} [videoKey] Custom video identifier
     * @param {boolean} [startEvent] Internal param informing that current interaction is responsible of first player start
     * @note Apparently, VideoAnalyticsRequestHandler queues requests until all mandatory events
     * (INIT / START) are sent. Then, it sends all the queued events. This is why there is no need
     * of queueing events here.
     */
    trackPlayerInteraction(eventName: string, dimensions?: object, metrics?: object, videoKey?: string): void;
    /**
     * Handle player event (queueing event in case video has not started yet or flushing the event
     * queue in case video starts)
     */
    private handlePlayerEvent;
    /**
     * Track player event
     *
     * @param eventName The name of the interaction (i.e., Pause, Seek, Skip Intro, Skip Ads, Switch
     *   Language, etc.).
     * @param contentID The unique content identifier of the content being played.
     * @param dimensions Dimensions to track
     * @param metrics Metrics to track
     * @param videoKey Video adapter identifier
     */
    private trackPlayerEvent;
    /**
     * Tracks search query events.
     * @param {string} searchQuery The search term entered by the user.
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    trackSearchQuery(searchQuery: string, dimensions?: object, metrics?: object): void;
    /**
     * Tracks search result events.
     * @param {integer} resultCount The number of search results returned by a search query.
     * @param {String} [searchQuery] The search term entered by the user.
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    trackSearchResult(resultCount: number, searchQuery?: string, dimensions?: object, metrics?: object): void;
    /**
     * Tracks user interactions with search results.
     * @param {string} section The section title. It is commonly used to indicate the section title presented over a grid layout (e.g. Recommended Movies, Continue Watching, etc).
     * @param {Number} sectionOrder The section order within the page.
     * @param {integer} column The content placement column. It is commonly used to indicate the column number where content is placed in a grid layout (i.e.1, 2, etc..).
     * @param {integer} row The content placement row. It is commonly used to indicate the row number where content is placed in a grid layout (i.e.1, 2, etc..).
     * @param {string} contentID The content identifier. It is used for internal content unequivocally identification (i.e., AAA000111222).
     * @param {String} [searchQuery] The search term entered by the user.
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    trackSearchClick(section: string | null, sectionOrder: number | null, column: number, row: number, contentID: string, searchQuery?: string, dimensions?: object, metrics?: object): void;
    /**
     * Tracks external app start events.
     * @param {string} appName The name of the application being used to deliver the content to the end-user (i.e., Netflix).
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    trackExternalAppLaunch(appName: string, dimensions?: object, metrics?: object): void;
    /**
     * Tracks external app stop events.
     * @param {string} appName The name of the application being used to deliver the content to the end-user (i.e., Netflix).
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    trackExternalAppExit(appName: string, dimensions?: object, metrics?: object): void;
    /**
     * Tracks engagement events.
     * @param {string} eventName The name of the engagement event (i.e., Share, Save, Rate, etc.).
     * @param {string} contentID The unique content identifier of the content the user is engaging with.
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    trackEngagementEvent(eventName: string, contentID: string, dimensions?: object, metrics?: object): void;
    /**
     * Track custom event
     * @param {string} eventName Name of the event to track
     * @param {Object} [dimensions] Dimensions to track
     * @param {Object} [metrics] Metrics to track
     */
    trackEvent(eventName: string, dimensions?: object, metrics?: object): void;
    /**
     * Fires an event
     * @param {string} eventName Name of the event to be fired
     * @param {ProductAnalyticsEventType} eventType Type of the event being tracked
     * @param {Object} dimensionsInternal Dimensions supplied by user
     * @param {Object} dimensionsUser Specific event dimensions
     * @param {Object} metrics Metrics to track
     * @private
     */
    private fireEventInternal;
    /**
     * Fires an adapter event (in case it is available)
     * @param {string} eventName Event name
     * @param {Object} dimensionsInternal Dimensions supplied by user
     * @param {Object} dimensionsUser Specific event dimensions
     * @param {Object} metrics Metrics to track
     * @param {string} [videoKey] Custom video identifier
     * @private
     */
    private fireEventAdapter;
    /**
     * Builds a list of top level and custom dimensions
     * @param {ProductAnalyticsEventType} eventType Type of the event being tracked
     * @param {Object} dimensionsInternal Object containing list of internal dimensions
     * @param {Object} dimensionsUser Object containing list of custom dimensions
     * @private
     */
    private buildDimensions;
    /**
     * Check state before sending an event
     * @param {String} message Message to show in case validation fails
     * @private
     */
    private checkState;
    private setListeners;
    private beforeStartListener;
    private afterStartListener;
    private beforeStopListener;
    private afterStopListener;
    private beforeSessionEventListener;
    private afterSessionEventListener;
}
