// Extract TTML tracks from manifest using Shaka Player
export async function extractTTMLTracksWithShaka(manifestUrl) {
    return new Promise(async (resolve, reject) => {
      try {
        // Check if Shaka Player is available
        if (typeof shaka === "undefined") {
          reject(new Error("Shaka Player is not available"));
          return;
        }
  
        const tempVideo = document.createElement("video");
        const shakaPlayer = new shaka.Player(tempVideo);
  
        await shakaPlayer.load(manifestUrl);
        const textTracks = shakaPlayer.getTextTracks();
  
        const ttmlTracks = textTracks
          .filter((t) => {
            // Check multiple conditions for TTML detection
            return (
              t.mimeType.includes("ttml") || // Direct TTML mime type
              (t.mimeType === "application/mp4" &&
                t.codecs &&
                t.codecs.includes("ttml")) || // MP4 with TTML codec
              (t.codecs && t.codecs.includes("stpp.ttml")) || // DASH TTML codec
              (t.kind === "subtitles" && t.codecs && t.codecs.includes("ttml")) // Subtitle with TTML codec
            );
          })
          .map((t) => ({
            src: t.uri || null, // URI may be undefined for embedded tracks
            language: t.language,
            label: t.label || `TTML ${t.language || "Unknown"}`,
            default: t.language === "en", // Default to English instead of Japanese
            type: t.mimeType || "application/ttml+xml",
            codecs: t.codecs,
            // For embedded tracks, we need different approach
            isEmbedded: !t.uri, // Flag to indicate embedded tracks
            originalTrack: t, // Keep reference to original track
          }));
  
        // Cleanup
        shakaPlayer.destroy();
  
        resolve(ttmlTracks);
      } catch (e) {
        reject(e);
      }
    });
  }
  
  // Check if URL is a manifest file (MPD/M3U8) or MP4
  export function isManifestFile(url) {
    const extension = url.split(".").pop().toLowerCase();
    return extension === "mpd" || extension === "m3u8";
  }
  
  // Load external TTML files for MP4 videos
  export async function loadExternalTTMLTracks(videoUrl, ttmlUrls) {
    return new Promise((resolve, reject) => {
      try {
        if (!Array.isArray(ttmlUrls)) {
          ttmlUrls = [ttmlUrls];
        }
  
        const tracks = ttmlUrls.map((ttmlUrl, index) => {
          // Extract language from URL or use default
          const language = extractLanguageFromUrl(ttmlUrl) || "en";
  
          return {
            src: ttmlUrl,
            language: language,
            label: `TTML ${language.toUpperCase()}`,
            default: index === 0, // First track is default
            type: "application/ttml+xml",
          };
        });
  
        resolve(tracks);
      } catch (e) {
        console.error("Failed to load external TTML tracks:", e);
        reject(e);
      }
    });
  }
  
  // Extract language from TTML URL (e.g., subtitle_en.ttml -> en)
  function extractLanguageFromUrl(url) {
    const matches = url.match(
      /[_-]([a-z]{2})[_.-]|\/([a-z]{2})[_.-]|\.([a-z]{2})\./i
    );
    return matches
      ? (matches[1] || matches[2] || matches[3]).toLowerCase()
      : null;
  }
  
  // Main function to handle both manifest and MP4 files
  export async function extractTTMLTracks(videoUrl, options = {}) {
    try {
      const isManifest = isManifestFile(videoUrl);
  
      if (isManifest) {
        // Handle manifest files (MPD/M3U8) with Shaka
        return await extractTTMLTracksWithShaka(videoUrl);
      } else {
        // Handle MP4 files with external TTML tracks
        if (options.ttmlUrls && options.ttmlUrls.length > 0) {
          return await loadExternalTTMLTracks(videoUrl, options.ttmlUrls);
        } else {
          console.warn("❌ No TTML URLs provided for MP4 video");
          return [];
        }
      }
    } catch (error) {
      console.error("❌ Error extracting TTML tracks:", error);
      return [];
    }
  }