import videojs from 'video.js';
import '../../../styles/time-display.css';

const Component = videojs.getComponent('Component');

class TimeDisplay extends Component {
  constructor(player, options) {
    super(player, options);
  }

  createEl() {
    const el = super.createEl('div', {
      className: 'videojs-drm-time-display'
    });
    
    el.innerHTML = `
      <span class="current-time">00:00</span> / <span class="duration">00:00</span>
    `;
    
    return el;
  }

  ready() {
    super.ready();
    this.setupTimeDisplay();
  }

  setupTimeDisplay() {
    const currentTimeSpan = this.el().querySelector('.current-time');
    const durationSpan = this.el().querySelector('.duration');
    
    // Update time display
    const updateTime = () => {
      const currentTime = this.formatTime(this.player().currentTime() || 0);
      const duration = this.formatTime(this.player().duration() || 0);
      
      currentTimeSpan.textContent = currentTime;
      durationSpan.textContent = duration;
    };

    // Listen to player events
    this.player().on('timeupdate', updateTime);
    this.player().on('loadedmetadata', updateTime);
    // this.player().tech().on('durationchange', () => {
    //   // console.log('durationchange1', this.player().duration(), this.player()?.tech()?.shaka_?.getManifest()?.presentationTimeline?.getDuration());
    //   updateTime();
    // });
    this.player().on('durationchange', () => {
      // console.log('durationchange', this.player().duration(), this.player()?.tech()?.shaka_?.getManifest()?.presentationTimeline?.getDuration());
      updateTime();
    });
  }

  formatTime(seconds) {
    if (isNaN(seconds) || !isFinite(seconds)) {
      return '00:00';
    }
    
    const hours = Math.floor(seconds / 3600);
    const minutes = Math.floor((seconds % 3600) / 60);
    const secs = Math.floor(seconds % 60);
    
    if (hours > 0) {
      return `${hours.toString().padStart(2, '0')}:${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
    } else {
      return `${minutes.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
    }
  }
}

// Register the component
videojs.registerComponent('TimeDisplay', TimeDisplay);

export default TimeDisplay;



