import videojs from "video.js";
import "video.js/dist/video-js.css";
import { setupDRM } from "./drm-config";
import CustomControlBar from "./components/CustomControlBar";
import Shaka from "./shaka/shaka.js";
import "./styles/styles.css";
import {
  DISPLAY_FORMAT,
  DISPLAY_FORMAT_VALUES,
} from "./constants/displayFormat.js";
import { detectSourceType, isAudioMime } from "./helpers/helper.js";

// Define default values for the plugin's `state` object here.
Shaka.defaultState = {};

// Register tech with Video.js
videojs.registerTech("shaka", Shaka);

class VideoJSDRMPlayer {
  constructor(options = {}) {
    // Validate required parameters
    if (!options.containerId) {
      throw new Error("containerId parameter is required");
    }

    if (!options.source) {
      throw new Error("source parameter is required");
    }

    const srcType = detectSourceType(options.source);
    this.isAudioOnly = isAudioMime(srcType);

    this.options = {
      // Default options
      totalWidth: "100%",
      totalHeight: this.isAudioOnly && options.controls ? "" : "auto",
      isDRM: false,
      useShakaTech: false,
      displayFormat: DISPLAY_FORMAT.DEFAULT,
      // Merge with user options
      ...options,
    };

    this.containerId = options.containerId;
    this.source = options.source;
    this.isDRM = options.isDRM ?? false;
    this.player = null;

    // Create video element
    this.createVideoElement();

    // Initialize Video.js player
    this.initializePlayer();
  }

  createVideoElement() {
    const containerElement = document.getElementById(this.containerId);
    if (!containerElement) {
      throw new Error(
        `Container element with id "${this.containerId}" not found`
      );
    }

    // Check if drmConfig is required
    if (
      this.options.isDRM &&
      (!this.options.drmConfig || !Object.keys(this.options.drmConfig).length)
    ) {
      throw new Error("drmConfig parameter is required if isDRM is true");
    }

    // Configure the existing video element
    // Create video element inside container
    const videoElement = document.createElement("video");
    videoElement.id = `${this.containerId}-video`;
    videoElement.className = "video-js vjs-default-skin";
    videoElement.setAttribute("playsinline", true);
    videoElement.setAttribute("webkit-playsinline", true);
    videoElement.setAttribute("x-webkit-airplay", "allow");

    if (this.options.totalWidth) {
      videoElement.style.width = this.options.totalWidth;
    }
    if (this.options.totalHeight) {
      videoElement.style.height = this.options.totalHeight;
    }

    // Clear container and add video element
    containerElement.innerHTML = "";
    containerElement.appendChild(videoElement);

    this.dispose();

    this.videoElement = videoElement;

    if (this.isAudioOnly) {
      if (this.options.controls) {
        this.options.audioOnlyMode = true;
      } else {
        this.updateAudioOnlyBeforeInitialization(true);
      }
    }
  }

  initializePlayer() {
    const playerOptions = {
      ...this.getVideoJSOptions(),
      techOrder: this.options.useShakaTech ? ["shaka", "html5"] : ["html5"],
    };

    this.player = videojs(this.videoElement, playerOptions);

    if (!this.player) {
      throw new Error("Failed to initialize Video.js player");
    }

    if (this.options.fluid) {
      this.videoElement.classList.add("vjs-fluid");
      this.player.fluid(true);
    }

    window.addEventListener("beforeunload", function () {
      this.dispose();
    });

    // Add error handling
    this.player.on("error", (error) => {
      console.error("Video.js Player Error:", error);
      this.emit("error", error);
    });

    // Add ready event
    this.player.ready(() => {
      console.log("Video.js Player is ready");

      // Setup DRM if needed
      if (!this.options.useShakaTech && this.isDRM) {
        setupDRM(
          this.player,
          {
            url: this.source,
            type: detectSourceType(this.source),
          },
          this.options.drmConfig
        );
      } else {
        this.player.src({
          src: this.source,
          type: detectSourceType(this.source),
        });
      }

      this.emit("ready");

      if (!this.options.controls) {
        const customControlBar = new CustomControlBar(this.player, { isAudioOnly: this.isAudioOnly });
        this.player.el().appendChild(customControlBar.el());
      }

      // Detect audio-only after metadata is available
      this.player.on("loadedmetadata", () => {
        // Wait for 500ms to ensure metadata is loaded (for Safari)
        setTimeout(() => {
          if (!this.options.controls) {
            this.updateAudioOnlyAfterInitialization();
          }
        }, 500);
      });

      // Also re-check on source changes
      // this.player.on("loadstart", () => {
      //   // remove previous state until metadata loads
      //   this.updateAudioOnlyClass(false);
      // });
    });
  }

  getVideoJSOptions() {
    const videojsOptions = { ...this.options };

    // Remove our custom options
    delete videojsOptions.containerId;
    delete videojsOptions.source;
    delete videojsOptions.isDRM;
    delete videojsOptions.totalWidth;
    delete videojsOptions.totalHeight;
    delete videojsOptions.displayFormat;
    if (this.options.useShakaTech && this.options.isDRM) {
      videojsOptions.shaka = {
        drmConfig: this.options.drmConfig,
      };
    } else {
      delete videojsOptions.drmConfig;
    }

    return videojsOptions;
  }

  updateAudioOnlyAfterInitialization() {
    try {
      // Check intrinsic video dimensions from tech
      if (!this.isAudioOnly && typeof this.player?.videoWidth === "function") {
        const vw = this.player.videoWidth();
        const vh = this.player.videoHeight();
        if ((vw === 0 && vh === 0) || Number.isNaN(vw) || Number.isNaN(vh)) {
          this.isAudioOnly = true;
          this.updateAudioOnlyBeforeInitialization(true);
        }
      }

      if (
        this.isAudioOnly &&
        !this.options.controls &&
        this.options.displayFormat === DISPLAY_FORMAT.PLAY_PAUSE_ONLY &&
        this.options.poster
      ) {
        const playButton = this.player
          .el()
          .querySelector(".videojs-drm-play-button");
        playButton.style.background = null;
        playButton.style.backgroundRepeat = "no-repeat";
        playButton.style.backgroundPosition = "center";
        playButton.style.backgroundSize = "cover";
        playButton.style.backgroundImage = `url(${this.options.poster})`;
      }
    } catch (e) {
      console.error("Error applying audio-only mode:", e);
    }
  }

  updateAudioOnlyBeforeInitialization(isAudioOnly) {
    if (!this.videoElement) return;

    // Remove all existing audio-only classes
    this.videoElement.classList.remove(
      "audio-only",
      "display-format-1",
      "display-format-2",
      "display-format-3"
    );

    if (isAudioOnly) {
      this.videoElement.classList.add("audio-only");

      if (DISPLAY_FORMAT_VALUES.includes(this.options.displayFormat)) {
        this.videoElement.classList.add(
          `display-format-${this.options.displayFormat}`
        );

        if (this.options.displayFormat === DISPLAY_FORMAT.HIDDEN) {
          this.videoElement.style.display = "none";
        }
      } else {
        console.warn(`Unknown displayFormat: ${this.options.displayFormat}`);
      }
    }
  }

  // Custom event system
  emit(eventName, data = null) {
    const event = new CustomEvent(
      `${this.containerId}-videojs-drm-player:${eventName}`,
      {
        detail: data,
      }
    );
    document.dispatchEvent(event);
  }

  on(eventName, callback) {
    document.addEventListener(
      `${this.containerId}-videojs-drm-player:${eventName}`,
      callback
    );
  }

  // Public API methods
  play() {
    if (this.player) {
      return this.player.play();
    }
  }

  pause() {
    if (this.player) {
      this.player.pause();
    }
  }

  stop() {
    if (this.player) {
      this.player.pause();
      this.player.currentTime(0);
    }
  }

  rewind(seconds = 10) {
    if (this.player) {
      const currentTime = this.player.currentTime();
      this.player.currentTime(Math.max(0, currentTime - seconds));
    }
  }

  forward(seconds = 10) {
    if (this.player) {
      const currentTime = this.player.currentTime();
      const duration = this.player.duration();
      this.player.currentTime(Math.min(duration, currentTime + seconds));
    }
  }

  setVolume(volume) {
    if (this.player && volume >= 0 && volume <= 1) {
      this.player.volume(volume);
    }
  }

  getVolume() {
    return this.player ? this.player.volume() : 0;
  }

  mute() {
    if (this.player) {
      this.player.muted(true);
    }
  }

  unmute() {
    if (this.player) {
      this.player.muted(false);
    }
  }

  toggleMute() {
    if (this.player) {
      this.player.muted(!this.player.muted());
    }
  }

  getCurrentTime() {
    return this.player ? this.player.currentTime() : 0;
  }

  getDuration() {
    return this.player ? this.player.duration() : 0;
  }

  seek(time) {
    if (this.player && time >= 0) {
      this.player.currentTime(time);
    }
  }

  setSize(width, height) {
    if (this.player) {
      this.player.dimensions(width, height);
      this.options.width = width;
      this.options.height = height;
    }
  }

  // Cleanup method
  dispose() {
    if (this.player && !this.player.isDisposed()) {
      this.player.dispose();
      this.player = null;
    }

    // Don't remove the video element since it's provided by the user
    // Just reset the reference
    this.videoElement = null;
  }
}

// Export for different module systems
export default VideoJSDRMPlayer;

// Also make it available globally if not using modules
if (typeof window !== "undefined") {
  window.VideoJSDRMPlayer = VideoJSDRMPlayer;
}
