# VideoJS DRM Player

A Video.js package library with integrated DRM and custom UI controls, supporting DRMtoday and advanced media player features.

## Features

- ✅ Fully integrated with Video.js
- ✅ DRM support with DRMtoday (Widevine, PlayReady, FairPlay)
- ✅ Custom UI controls (play/pause/stop/rewind/forward/volume)
- ✅ Responsive design with automatic video sizing
- ✅ Video content always matches the size of the media player
- ✅ Flexible configuration
- ✅ Error handling and logging

## Installation

```bash
npm install videojs-drm-player
```

## Install dependencies

This library requires Video.js and videojs-contrib-eme:

```bash
npm install video.js videojs-contrib-eme
```

## Basic Usage

### HTML

```html
<!DOCTYPE html>
<html>
<head>
    <link href="https://vjs.zencdn.net/8.5.2/video-js.css" rel="stylesheet">
</head>
<body>
    <div id="my-video-container"></div>
    
    <script src="https://vjs.zencdn.net/8.5.2/video.min.js"></script>
    <script src="path/to/videojs-drm-player.min.js"></script>
</body>
</html>
```

### JavaScript

```javascript
import VideoJSDRMPlayer from 'videojs-drm-player';

// Initialize player with a regular video
const player = new VideoJSDRMPlayer({
    container_id: 'my-video-container',
    source: 'https://example.com/video.mp4',
    width: 800,
    height: 450,
    autoplay: false,
    controls: true
});
```

### Usage with DRM

```javascript
import VideoJSDRMPlayer, { createDRMtodayConfig } from 'videojs-drm-player';

// DRMtoday configuration
const drmConfig = createDRMtodayConfig(
    'your-merchant-id',
    'user-id',
    'session-id',
    { customField: 'value' }
);

// Initialize player with DRM
const player = new VideoJSDRMPlayer({
    container_id: 'my-video-container',
    source: 'https://example.com/drm-protected-video.mpd',
    isDRM: true,
    drmConfig: drmConfig,
    width: 1280,
    height: 720,
    autoplay: false,
    controls: true
});
```

## API Reference

### Constructor Options

| Parameter        | Type    | Required | Default | Description                       |
|-----------------|---------|----------|---------|-----------------------------------|
| `container_id`   | string  | ✅       | -       | ID of the container element       |
| `source`         | string  | ✅       | -       | Video source URL                  |
| `isDRM`          | boolean | ❌       | false   | Enable DRM                        |
| `width`          | number  | ❌       | 100     | Player width (px)                 |
| `height`         | number  | ❌       | 100     | Player height (px)                |
| `autoplay`       | boolean | ❌       | false   | Autoplay                          |
| `loop`           | boolean | ❌       | false   | Loop playback                     |
| `muted`          | boolean | ❌       | false   | Mute                              |
| `controls`       | boolean | ❌       | true    | Show controls                     |
| `fluid`          | boolean | ❌       | false   | Responsive fluid                   |
| `drmConfig`      | object  | ❌       | -       | DRM configuration                 |

### Public Methods

#### Playback Control

```javascript
// Play video
player.play();

// Pause
player.pause();

// Stop and reset to start
player.stop();

// Rewind (default 10 seconds)
player.rewind(10);

// Forward (default 10 seconds)
player.forward(15);

// Seek to a specific time
player.seek(60); // Jump to 60th second
```

#### Volume Control

```javascript
// Set volume (0.0 - 1.0)
player.setVolume(0.5);

// Get current volume
const volume = player.getVolume();

// Mute
player.mute();

// Unmute
player.unmute();

// Toggle mute/unmute
player.toggleMute();
```

#### Information

```javascript
// Get current time
const currentTime = player.getCurrentTime();

// Get total duration
const duration = player.getDuration();
```

#### Player Management

```javascript
// Change size
player.setSize(1920, 1080);

// Get current aspect ratio
const currentRatio = player.getAspectRatio();

// Dispose player
player.dispose();
```

### Events

```javascript
// Listen for player ready event
player.on('ready', (event) => {
    console.log('Player is ready!');
});

// Listen for errors
player.on('error', (event) => {
    console.error('Player error:', event.detail);
});

// Listen for DRM errors
document.addEventListener('videojs-drm-player:drm-error', (event) => {
    console.error('DRM Error:', event.detail);
});
```

## DRM Configuration

### DRMtoday Integration

```javascript
import { createDRMtodayConfig } from 'videojs-drm-player';

const drmConfig = createDRMtodayConfig(
    'merchant-123',
    'user-456', 
    'session-789'
);
```

### Custom DRM Configuration

```javascript
const customDRMConfig = {
    widevine: {
        url: 'https://your-license-server.com/widevine',
        headers: {
            'Authorization': 'Bearer your-token',
            'Content-Type': 'application/octet-stream'
        }
    },
    playready: {
        url: 'https://your-license-server.com/playready',
        headers: {
            'Authorization': 'Bearer your-token',
            'Content-Type': 'text/xml; charset=utf-8'
        }
    }
};

const player = new VideoJSDRMPlayer({
    container_id: 'video-container',
    source: 'https://example.com/protected-video.mpd',
    isDRM: true,
    drmConfig: customDRMConfig
});
```

## Custom Controls

The library comes with custom controls:

- **Rewind Button**: Rewind 10 seconds
- **Forward Button**: Forward 10 seconds
- **Stop Button**: Stop and reset to start
- **Custom Volume Panel**: Fine-grained volume control with slider

## Responsive Design

The player automatically adapts to different screen sizes:

- **Desktop**: Full controls displayed
- **Tablet**: Volume value text hidden
- **Mobile**: Volume slider hidden, only mute button shown

## Development

### Build from source

```bash
# Clone repository
git clone https://github.com/yourusername/videojs-drm-player.git
cd videojs-drm-player

# Install dependencies
npm install

# Build production
npm run build

# Development with watch mode
npm run dev
```

## License

MIT License - see [LICENSE](LICENSE) for details.
