import videojs from "video.js";
import "./SettingsPopup.css";
import {
  formatQualityLabel,
  sortQualityTracks,
} from "../../../helpers/renditionHelper";
import {
  playbackSpeedIcon,
  qualityIcon,
  subtitleIcon,
  chevronLeftIcon,
  checkIcon,
} from "../../../assets/icons/index.js";

const Component = videojs.getComponent("Component");

const AUTO_QUALITY_ID = -1;
const DEFAULT_QUALITY = {
  id: AUTO_QUALITY_ID,
  label: "Auto",
};

class SettingsPopup extends Component {
  constructor(player, options) {
    super(player, options);
    this.isPopupOpen = false;
    this.currentSpeed = 1;
    this.currentRendition = DEFAULT_QUALITY;
    this.renditions = [];
    this.currentSubtitle = null;
    this.subtitles = [];
    this.currentMenu = "main"; // 'main', 'playback-speed', 'quality', or 'subtitles'
  }

  createEl() {
    const el = super.createEl("div", {
      className: "videojs-drm-settings-popup",
    });
    el.style.display = "none";

    return el;
  }

  ready() {
    super.ready();

    this.isShakaPlayer =
      this.player().options().useShakaTech &&
      this.player().techName_ === "Shaka";
    // Create main menu content
    this.createMainMenu();

    this.attachEvents();
    this.setupRenditionListener();
    this.setupSubtitleListener();
  }

  // Main popup container
  createMainMenu() {
    const mainMenuOptions = [
      {
        action: "playback-speed",
        icon: playbackSpeedIcon,
        title: "Playback speed",
        value: this.currentSpeed === 1 ? "Normal" : this.currentSpeed + "x",
      },
      {
        action: "quality",
        icon: qualityIcon,
        title: "Quality",
        value: this.currentRendition?.label,
      },
      {
        action: "subtitles",
        icon: subtitleIcon,
        title: "Subtitles",
        value: this.currentSubtitle
          ? this.currentSubtitle.label || this.currentSubtitle.language || "Unknown"
          : "Off",
      },
    ];

    this.mainMenuContent = mainMenuOptions
      .map(
        (option) => `
        <div class="vjsd-main-menu settings-popup-item" data-action="${option.action}">
            ${option.icon}
            <span class="settings-text settings-title">${option.title}</span>
            <span class="settings-value">${option.value}</span>
        </div>
    `
      )
      .join("");
  }

  createPlaybackSpeedMenu() {
    const speedOptions = [
      { value: 0.5, label: "0.5x", selected: this.currentSpeed === 0.5 },
      { value: 0.75, label: "0.75x", selected: this.currentSpeed === 0.75 },
      { value: 1, label: "Normal", selected: this.currentSpeed === 1 },
      { value: 1.25, label: "1.25x", selected: this.currentSpeed === 1.25 },
      { value: 1.5, label: "1.5x", selected: this.currentSpeed === 1.5 },
      { value: 2, label: "2x", selected: this.currentSpeed === 2 },
    ];

    this.playbackSpeedMenuContent = this.createSubmenu(
      "Playback speed",
      "speed",
      speedOptions,
      ["speed-item"]
    );
  }

  createQualityMenu() {
    // Add Auto option
    const qualityOptions = [];
    if (!this.renditions.some((r) => r.id == AUTO_QUALITY_ID)) {
      qualityOptions.push({
        value: AUTO_QUALITY_ID,
        label: "Auto",
        selected: this.currentRendition?.id == AUTO_QUALITY_ID,
      });
    }

    // Add available renditions
    this.renditions.forEach((rendition) => {
      qualityOptions.push({
        value: rendition.id,
        label: rendition.label,
        selected: this.currentRendition?.id == rendition.id,
      });
    });

    this.qualityMenuContent = this.createSubmenu(
      "Quality",
      "quality",
      qualityOptions,
      ["quality-item"]
    );
  }

  createSubtitlesMenu() {
    // Add Off option
    const subtitleOptions = [{
        value: 'off',
        label: 'Off',
        selected: !this.currentSubtitle || this.currentSubtitle.id === 'off',
    }];

    // Add available subtitles
    this.subtitles.forEach((subtitle) => {
      subtitleOptions.push({
        value: subtitle.id,
        label: subtitle.label || subtitle.language || "Unknown",
        selected: this.currentSubtitle?.id == subtitle.id,
      });
    });

    this.subtitlesMenuContent = this.createSubmenu(
      "Subtitles",
      "subtitle",
      subtitleOptions,
      ["subtitle-item"]
    );
  }

  createSubmenu(title, action, options, classNames) {
    return `
      ${this.createBackButton(title)}
      ${options
        .map(
          (option) => `
        <div class="settings-popup-item ${classNames.join(" ")} ${
            option.selected ? "selected" : ""
          }" data-${action}="${option.value}">
            <span class="settings-text">${option.label}</span>
            ${option.selected ? checkIcon : ""}
        </div>
        `
        )
        .join("")}
    `;
  }

  createBackButton(title) {
    return `
      <div class="settings-popup-item back-button" data-action="back">
        ${chevronLeftIcon}
        <span class="settings-text settings-title">${title}</span>
      </div>
    `;
  }

  attachEvents() {
    // Use event delegation for dynamic content
    const handleInteraction = (e) => {
      e.stopPropagation();

      const item = e.target.closest(".settings-popup-item");
      if (!item) return;

      const action = item.dataset.action;
      const speed = item.dataset.speed;
      const quality = item.dataset.quality;
      const subtitle = item.dataset.subtitle;

      if (action === "playback-speed") {
        this.showPlaybackSpeedMenu();
      } else if (action === "quality") {
        this.showQualityMenu();
      } else if (action === "subtitles") {
        this.showSubtitlesMenu();
      } else if (action === "back") {
        this.showMainMenu();
      } else if (speed) {
        this.setPlaybackSpeed(parseFloat(speed));
      } else if (quality) {
        this.setQuality(quality);
      } else if (subtitle) {
        this.setSubtitle(subtitle);
      }
    };

    // Add both click and touch events for better mobile support
    this.el().addEventListener("click", handleInteraction);
    this.el().addEventListener("touchend", (e) => {
      // Prevent click event from firing after touchend
      e.preventDefault();
      handleInteraction(e);
    });
  }

  toggle() {
    this.isPopupOpen = !this.isPopupOpen;
    if (this.isPopupOpen) {
      this.show();
    } else {
      this.hide();
    }
  }

  show() {
    this.el().style.display = "block";
    this.isPopupOpen = true;
    this.currentMenu = "main";

    this.createMainMenu();
    this.showMainMenu();

    setTimeout(() => {
      this.outsideHandler = (evt) => {
        if (!this.el().contains(evt.target)) {
          this.hide();
        }
      };
      document.addEventListener("click", this.outsideHandler);
    }, 0);
  }

  hide() {
    this.el().style.display = "none";
    this.isPopupOpen = false;
    this.currentMenu = "main";

    if (this.outsideHandler) {
      document.removeEventListener("click", this.outsideHandler);
      this.outsideHandler = null;
    }
  }

  showMainMenu() {
    this.currentMenu = "main";
    this.el().innerHTML = this.mainMenuContent;
  }

  showPlaybackSpeedMenu() {
    this.currentMenu = "playback-speed";
    // Ensure content is up to date
    this.createPlaybackSpeedMenu();
    this.el().innerHTML = this.playbackSpeedMenuContent;
  }

  showQualityMenu() {
    this.currentMenu = "quality";
    // Ensure content is up to date
    this.createQualityMenu();
    this.el().innerHTML = this.qualityMenuContent;
  }

  showSubtitlesMenu() {
    this.currentMenu = "subtitles";
    // Ensure content is up to date
    this.createSubtitlesMenu();
    this.el().innerHTML = this.subtitlesMenuContent;
  }

  setPlaybackSpeed(speed) {
    this.currentSpeed = speed;
    if (this.player) {
      this.player().playbackRate(speed);
    }
    // Close popup after selection
    this.hide();
  }

  setQuality(qualityId) {
    try {
      const rendition = this.renditions.find((r) => r.id == qualityId) || null;
      if (!rendition) return;

      this.currentRendition = rendition;
      if (this.isShakaPlayer && this.qualitySwitchCallback) {
        this.qualitySwitchCallback(rendition.id, "video");
        return;
      }

      this.setRendition(rendition);
    } catch (error) {
      console.warn("Could not set quality:", error);
    } finally {
      this.hide();
    }
  }

  setSubtitle(subtitleId) {
    if (subtitleId === "off") {
      this.currentSubtitle = null;
      this.disableSubtitles();
    } else {
      const subtitle = this.subtitles.find((s) => s.id == subtitleId);
      if (subtitle) {
        this.currentSubtitle = subtitle;
        this.enableSubtitle(subtitle);
      }
    }
    // Close popup after selection
    this.hide();
  }

  setupRenditionListener() {
    if (!this.player()) return;

    const tech = this.player().tech();
    if (tech && this.isShakaPlayer) {
      tech.on(
        "loadedqualitydata",
        (event, { qualityData, qualitySwitchCallback }) => {
          this.qualitySwitchCallback = qualitySwitchCallback;
          this.updateRenditionsFromShaka(qualityData);
        }
      );
    }

    this.player().on("qualitytrackchange", (event, options) => {
      this.updateCurrentRendition(options);
    });

    if (!this.isShakaPlayer) {
      // Listen for loadedmetadata to get renditions
      this.player().on("loadedmetadata", () => {
        this.updateRenditions();
      });

      // Listen for source changes
      this.player().on("loadstart", () => {
        this.updateRenditions();
      });

      // Listen for quality changes
      this.player().on("qualitychange", () => {
        this.updateRenditions();
      });
    }
  }

  setupSubtitleListener() {
    if (!this.player) return;
    
    // this.player().tech().on("texttracksloaded", () => {
    //   this.updateSubtitles();
    // });

    // Listen for loadedmetadata to get subtitles
    this.player().on("loadedmetadata", () => {
      // Set timeout to ensure text tracks are loaded (for Safari)
      setTimeout(() => {
        this.updateSubtitles();
      }, 1000);
    });

    // Listen for subtitle changes from CCButton
    this.player().on('subtitlechanged', () => {
      this.updateCurrentSubtitle();
    });

    // Listen for source changes
    // this.player().on("loadstart", () => {
    //   this.updateSubtitles();
    // });

    // Listen for text track changes
    // this.player().on("texttrackchange", () => {
    //   this.updateCurrentSubtitle();
    // });
  }

  updateRenditionsFromShaka(qualityData) {
    try {
      if (qualityData?.video?.length > 0) {
        this.renditions = qualityData.video.map((track, index) => ({
          id: track.id,
          bandwidth: track.bandwidth,
          width: track.width,
          height: track.height,
          label: track.label,
          index: index,
          originalTrack: track,
        }));

        // Keep the order from quality data (Auto first, then by height, then by bandwidth)
        this.currentRendition =
          this.renditions.find((r) => r.id == AUTO_QUALITY_ID) || null;
      }
    } catch (error) {
      console.warn("Could not update renditions from quality data:", error);
    }
  }

  updateRenditions() {
    console.log("Updating renditions");
    try {
      // Try using qualityLevels API first
      if (
        this.player().qualityLevels &&
        this.player().qualityLevels()?.length > 0
      ) {
        this.renditions = Array.from(this.player().qualityLevels())
          .map((level, index) => ({
            id: `quality-${index}`,
            label: formatQualityLabel(level),
            bandwidth: level.bandwidth,
            width: level.width,
            height: level.height,
            index: index,
            qualityLevel: level,
          }))
          .sort(sortQualityTracks);

        console.log(
          "Available renditions from qualityLevels:",
          this.renditions
        );
      } else if (this.player().tech()?.vhs) {
        console.log("Quality levels is not usable, trying VHS...");
        const vhs = this.player().tech().vhs;

        if (vhs.representations && vhs.representations()) {
          const rawReps = vhs.representations();

          this.renditions = rawReps
            .map((rep, index) => ({
              id: `rendition-${index}`,
              label: formatQualityLabel(rep),
              bandwidth: rep.bandwidth,
              width: rep.width,
              height: rep.height,
              resolution: rep.resolution,
              index: index,
              representation: rep,
            }))
            .sort(sortQualityTracks);

          console.log("Available renditions from VHS:", this.renditions);
        }
      }
    } catch (error) {
      console.warn("Could not get renditions:", error);
    } finally {
      if (!this.renditions.some((r) => r.id == AUTO_QUALITY_ID)) {
        this.renditions.unshift(DEFAULT_QUALITY);
      }
    }
  }

  updateCurrentRendition(options) {
    try {
      if (options && options.id !== undefined) {
        this.currentRendition =
          this.renditions.find((r) => r.id == options.id) || null;
        console.log("Current rendition updated to:", this.currentRendition);
      }
    } catch (error) {
      console.warn("Could not update current rendition:", error);
    }
  }

  setRendition(rendition = null) {
    try {
      const toAutoRendition = rendition.id === AUTO_QUALITY_ID;
      // Try using qualityLevels API first
      if (
        this.player().qualityLevels &&
        this.player().qualityLevels()?.length > 0
      ) {
        // Disable all quality levels first
        Array.from(this.player().qualityLevels()).forEach((level, index) => {
          level.enabled = toAutoRendition;
        });

        // Enable only the selected quality level
        if (rendition.qualityLevel) {
          rendition.qualityLevel.enabled = true;
          console.log(
            "Enabled quality level:",
            rendition.qualityLevel ?? rendition.label
          );
        }
      } else if (this.player().tech()?.vhs && rendition.representation) {
        const vhs = this.player().tech().vhs;
        const reps = vhs.representations();

        reps.forEach((rep) => {
          if (rep === rendition.representation) {
            rep.enabled = true;
          } else {
            rep.enabled = toAutoRendition;
          }
        });
        console.log("Set rendition to:", rendition);

        // Force VHS to update
        if (vhs.trigger) {
          vhs.trigger("renditionchange");
        }
      }
    } catch (error) {
      console.warn("Could not set rendition:", error);
    }
  }

  updateSubtitles() {
    try {
      let textTracks = this.player().textTracks();

      this.subtitles = Array.from(textTracks)
        .filter(
          (track) =>
            track.kind === "captions" ||
            track.kind === "subtitles" ||
            track.kind === "subtitle"
        )
        .map((track, index) => ({
          id: track.id || `subtitle-${index}`,
          language: track.language || "unknown",
          label: track.label || track.language || `Subtitle ${index + 1}`,
          kind: track.kind,
          index: index,
          textTrack: track,
        }));

      console.log("Available subtitles:", this.subtitles);

      // Update current subtitle based on active track
      this.updateCurrentSubtitle();
    } catch (error) {
      console.warn("Could not get subtitles:", error);
    }
  }

  updateCurrentSubtitle() {
    try {
      const textTracks = this.player().textTracks();
      const activeTrack = Array.from(textTracks).find(
        (track) =>
          track.mode === "showing" &&
          (track.kind === "captions" ||
            track.kind === "subtitles" ||
            track.kind === "subtitle")
      );

      if (activeTrack) {
        this.currentSubtitle = this.subtitles.find(
          (subtitle) => subtitle.textTrack === activeTrack
        );
      } else {
        this.currentSubtitle = null;
      }

      console.log("Current subtitle:", this.currentSubtitle);
    } catch (error) {
      console.warn("Could not update current subtitle:", error);
    }
  }

  enableSubtitle(subtitle) {
    try {
      console.log("Enabling subtitle:", subtitle);

      // Disable all text tracks first
      const textTracks = this.player().textTracks();
      Array.from(textTracks).forEach((track) => {
        track.mode = "disabled";
        if (track === subtitle.textTrack) {
          track.mode = "showing";
        }
      });

      this.updateCCButtonState();
    } catch (error) {
      console.warn("Could not enable subtitle:", error);
    }
  }

  disableSubtitles() {
    try {
      const textTracks = this.player().textTracks();
      Array.from(textTracks).forEach((track) => {
        if (track.kind === "captions" || track.kind === "subtitles" || track.kind === "subtitle") {
          track.mode = "disabled";
        }
      });

      this.updateCCButtonState();
    } catch (error) {
      console.warn("Could not disable subtitles:", error);
    }
  }
  
  updateCCButtonState() {
    // Use VideoJS events to communicate with CCButton
    this.player().trigger('subtitlechanged');
  }

  dispose() {
    super.dispose();
    this.el().remove();
  }
}

videojs.registerComponent("SettingsPopup", SettingsPopup);

export default SettingsPopup;
