import videojs from 'video.js';
import { subtitleToggleIcon } from "../../../assets/icons/index.js";
import './CCButton.css';

const Button = videojs.getComponent('Button');

class CCButton extends Button {
  constructor(player, options) {
    super(player, options);
    this.controlText('Closed Captions');
  }

  createEl() {
    const el = super.createEl('button', {
      className: 'videojs-drm-cc-button vjs-control vjs-button'
    });
    
    el.innerHTML = `
      ${subtitleToggleIcon}
      <div class="videojs-drm-subtitle-toggle-indicator"></div>
    `;

    return el;
  }

  handleClick() {
    // Toggle captions if available
    const tracks = this.player().textTracks();
    const captionTracks = Array.from(tracks).filter(track => 
      track.kind === 'captions' || track.kind === 'subtitles' || track.kind === "subtitle"
    );
    
    if (captionTracks.length === 0) {
      return; // No captions available
    }
    
    // Check if any caption is currently showing
    const hasActiveCaption = captionTracks.some(track => track.mode === 'showing');
    
    if (hasActiveCaption) {
      // Turn off all captions
      captionTracks.forEach(track => {
        track.mode = 'disabled';
      });
    } else {
      // Turn on the first caption track
      captionTracks[0].mode = 'showing';
    }
    
    // Update button state
    this.updateButtonState();
    
    // Notify settings popup if it exists
    this.notifySettingsPopup();
  }
  
  notifySettingsPopup() {
    // Use VideoJS events to communicate with SettingsPopup
    this.player().trigger('subtitlechanged');
  }

  ready() {
    super.ready();

    // Listen for loadedmetadata to get subtitles
    this.player().on("loadedmetadata", () => {
      // Set timeout to ensure text tracks are loaded (for Safari)
      this.setTimeout(() => {
        this.disableSubtitlesByDefault();
        this.updateButtonState();
      }, 1000);
    });
    
    // this.player().on('texttrackchange', () => {
    //   this.updateButtonState();
    // });
    
    // Listen for subtitle changes from SettingsPopup
    this.player().on('subtitlechanged', () => {
      this.updateButtonState();
    });
  }
  
  disableSubtitlesByDefault() {
    const tracks = this.player().textTracks();
    Array.from(tracks).filter(track => 
      track.kind === 'captions' || track.kind === 'subtitles' || track.kind === 'subtitle'
    ).forEach(track => {
      track.mode = 'disabled';
    });
  }

  updateButtonState() {
    const tracks = this.player().textTracks();
    const hasCaptions = Array.from(tracks).some(track => 
      track.kind === 'captions' || track.kind === 'subtitles' || track.kind === 'subtitle'
    );
    
    const hasActiveCaption = Array.from(tracks).some(track => 
      (track.kind === 'captions' || track.kind === 'subtitles' || track.kind === 'subtitle') && track.mode === 'showing'
    );

    const indicator = this.el().querySelector('.videojs-drm-subtitle-toggle-indicator');
    
    // Enable/disable button based on caption availability
    this.el().disabled = !hasCaptions;
    this.el().style.opacity = hasCaptions ? '1' : '0.5';
    
    // Add red underline when captions are active
    if (hasActiveCaption) {
      indicator.style.opacity = '1';
    } else {
      indicator.style.opacity = '0';
    }
  }
}

// Register the component
videojs.registerComponent('CCButton', CCButton);

export default CCButton;



